import db from "../db-config.js";
import {
  countQueryCondition,
  getRecord,
  makeJoins,
  searchConditionRecord,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import AuditLog from "../sequelize/AuditLogSchema.js";

/**Function to view all Audit Execution */
export const viewAllAuditLog = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "audit_log",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "audit_log.document_name",
    "audit_log.action_type",
    "audit_log.document_status", 
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = audit_log.created_by",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = audit_log.repository_id",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = audit_log.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_log.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const auditLogDataFetchQuery = `SELECT audit_log.*, repository.name as document_name,
  repository.url as document_file, department.name as department_name,
      CONCAT(users.name , ' ' , users.surname) as user_name, users.profile, audit_log.organization,organization.name as organization_name
      FROM audit_log  
      ${joinsRecord} 
      where audit_log.deleted = 0 ${searchCondition} ${condition}`;

  const [auditLogDataFetch] = await db.query(auditLogDataFetchQuery);

  /**Count all Audit Execution */
  const totalRecord = await countQueryCondition(auditLogDataFetchQuery);

  return sendResponse(res, 200, auditLogDataFetch, totalRecord);
};

export const getUserAndDocumentDropdown = async (req, res) => {
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = audit_log.created_by",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = audit_log.repository_id",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = audit_log.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_log.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  // Get unique users with id
  const userQuery = `
    SELECT DISTINCT 
      users.id AS user_id,
      CONCAT(users.name, ' ', users.surname) AS user_name
    FROM audit_log  
    ${joinsRecord}
    WHERE audit_log.deleted = 0 AND users.id IS NOT NULL
  `;

  // Get unique documents with id
  const documentQuery = `
    SELECT DISTINCT 
      repository.id AS document_id,
      repository.name AS document_name
    FROM audit_log  
    ${joinsRecord}
    WHERE audit_log.deleted = 0 AND repository.id IS NOT NULL AND repository.name IS NOT NULL
  `;

  const [userResult] = await db.query(userQuery);
  const [documentResult] = await db.query(documentQuery);

  return sendResponse(res, 200, {
    users: userResult,
    documents: documentResult,
  });
};

