import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  makeLoopAndGetMultipleUsersDetails,
  searchConditionRecord,
  storeError,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { createCalendar } from "../helper/calendar.js";
import { fileURLToPath } from "url";
import path from "path";
import fs from "fs";
import ContractorMeeting from "../sequelize/ContractorMeetingSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorMeetingRecording from "../sequelize/ContractorMeetingRecordingSchema.js";
// root directory path of the project
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

/**Function to create and update ContractorMeeting   */
export const createUpdateContractorMeeting = async (req, res) => {
  const { id, department } = req.body;
  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }

  const meetingId = Math.floor(1000 + Math.random() * 9000);

  req.body.meeting_id = id ? meetingId : req.body.meeting_id;

  let status = id ? "update" : "create";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ContractorMeeting",
      "contractor_meeting",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(ContractorMeeting, req.body)
    : createQueryBuilder(ContractorMeeting, req.body);
  const [result] = await db.query(query, values);

  if (!id) {
    const { query, values } = createQueryBuilder(ContractorMeetingRecording, {
      organization: req.body.organization,
      department: req.body.department,
      scheduled_meeting: result.insertId,
      status: "New",
      approval_status: "Pending",
    });
    await db.query(query, values);
  }

  return sendResponse(res, 200, `Record ${status} Successfully`);
};
/**Function to view all contractor meeting */
export const viewAllMeeting = async (req, res) => {
  const condition = await whereCondition({
    table: "contractor_meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });
  /** If value come with any search condition then search that word */
  const searchTableName = [
    "contractor_meeting.meeting_title",
    "contractor_meeting.venue",
    "contractor_meeting.meeting_hierarchy",
    "organization.name",
    "contractor_meeting.meeting_title",
    "contractor_meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);
  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = contractor_meeting.created_by",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = contractor_meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_meeting.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_meeting.department",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition: "contractor_registration.id = contractor_meeting.contractor",
    },
    {
      type: "left",
      targetTable: "project_registration",
      onCondition: "project_registration.id = contractor_meeting.project",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  /**Record of all meetings */
  const meetingsQuery = `SELECT contractor_meeting.*, contractor_meeting.created_by as created_by_id,CONCAT(createdUser.name , ' ' , createdUser.surname) AS created_by_name, createdUser.profile as created_by_profile ,CONCAT(u1.name , ' ' , u1.surname) AS meeting_owner_name ,u1.profile as meeting_owner_profile , organization.name as organization_name , contractor_registration.contractor_name AS contractor_name , contractor_registration.cipc_registration_number AS contractor_cipc_registration_number , project_registration.project_title AS project_name FROM contractor_meeting
     ${joinsRecord} WHERE contractor_meeting.deleted = 0 ${searchCondition} ${condition}`;
  let [meetings] = await db.query(meetingsQuery);
  meetings = await decodeAndParseFields(meetings);

  for (const contractor_meeting of meetings) {
    const agendas = contractor_meeting.agenda;
    for (const agenda of agendas) {
      const userId = agenda.allocated_to;
      let userName = "";
      if (userId) {
        [userName] = await db.query(
          `SELECT  CONCAT(name , ' ' , surname) AS name, profile FROM users WHERE id = ${userId}`
        );
      }
      agenda.allocated_name = userName[0]?.name;
      agenda.allocated_profile = userName[0]?.profile;
    }

    let arr = [];
    const externalParticipants = contractor_meeting.external_participants;
    for (const externalParticipant of externalParticipants) {
      const [user] = await db.query(
        `SELECT id , contractor_name AS name FROM contractor_registration WHERE id = ${externalParticipant}`
      );
      if (user[0]) {
        user[0].type = "External";
        arr.push(user[0]);
      }
    }
    contractor_meeting.external_participant_details = arr;
    arr = [];
    const participants = contractor_meeting.participants;
    for (const participant of participants) {
      const [user] = await db.query(
        `SELECT id ,CONCAT(users.name , ' ' ,users.surname) AS name , users.profile FROM users WHERE id = ${participant}`
      );
      user[0].type = "Internal";
      arr.push(user[0]);
    }
    contractor_meeting.participant_details = arr;

    contractor_meeting.attendees = [
      ...contractor_meeting.external_participant_details,
      ...contractor_meeting.participant_details,
    ];
  }
  const totalRecord = await countQueryCondition(meetingsQuery);

  return sendResponse(res, 200, meetings, totalRecord);
};

export const viewAllMeetingCalendar = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  let searchCondition = req.query.search ? `AND (contractor_meeting.meeting_title LIKE '%${req.query.search}%')` : "";
  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_meeting.created_by",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = contractor_meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_meeting.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_meeting.department",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  /**Record of all meetings */
  const meetingsQuery = `SELECT organization.name as organization_name, contractor_meeting.id,contractor_meeting.meeting_id,contractor_meeting.meeting_title,contractor_meeting.meeting_hierarchy,contractor_meeting.participants,contractor_meeting.meeting_owner,contractor_meeting.planned_meeting_date_from,contractor_meeting.planned_meeting_date_to,contractor_meeting.venue,contractor_meeting.agenda,contractor_meeting.organization,contractor_meeting.department,contractor_meeting.created_by as created_by_id,u1.name as host_name,users.name as created_by
     FROM contractor_meeting
     ${joinsRecord} where contractor_meeting.deleted = 0 ${searchCondition} ${condition}`;
  const meetings = await db.query(meetingsQuery);

  const fileObj = await createCalendar(meetings);
  const completePath = path.join(__dirname, "../", fileObj?.filePath);
  // Check if the file exists
  if (!fs.existsSync(completePath)) {
    return res.status(404).json({
      status: false,
      message: "File not found",
    });
  }
  return res.status(200).json({
    status: true,
    data: fileObj.fileName,
  });
};

/**Function to delete a specific ContractorMeeting */
export const deleteContractorMeeting = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("contractor_meeting", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "ContractorMeeting", id);
    return res.status(200).json({
      status: true,
      message: "Record deleted successfully",
    });
  }
};
