import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  whereCondition,
  settingsUpdater,
  searchConditionRecord,
  decodeAndParseFields,
} from "../helper/general.js";
import IncidentCategory from "../sequelize/IncidentCategorySchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new Incident Category for employee  */
export const createUpdateIncidentCategory = async (req, res) => {
  const { id, name, description, organization } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  /**If id comes in body then it will update the query */
  if (id) {
    if (organization.length === 0) {
      return sendResponse(
        res,
        200,
        "Please Select at least one business Structure"
      );
    }
    const result = await settingsUpdater(
      IncidentCategory,
      organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 200, "No data found");
    }

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Category Type",
      `This user Update Category Type which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    /** Insert new record */
    const insertPromises = organization.map(async (element) => {
      // Check if the category type already exists for the organization
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM incident_category WHERE name = ? AND deleted = "0" AND organization = ?`,
        [name, element]
      );

      // If record not found for that name with particular organization, insert the record
      if (checkNameWithOrganization.length === 0) {
        // Insert Category Type details
        const insertIncidentCategoryQuery = `INSERT INTO incident_category(name, description, created_by, organization) VALUES (?, ?, ?, ?)`;
        await db.query(insertIncidentCategoryQuery, [
          name,
          encodedDescription,
          req.user.sessionid,
          element,
        ]);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Category Type",
          `This user created a new category '${name}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Incident Category */
export const viewIncidentCategory = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "incident_category",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "incident_category.name",
    "incident_category.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = incident_category.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = incident_category.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const incidentCategoryQuery = `SELECT incident_category.id,incident_category.created_by as created_by_id,incident_category.name,incident_category.description,users.name as created_by,users.surname as created_by_surname ,users.profile as created_by_profile,organization.name as organization_name ,incident_category.organization FROM incident_category ${joinsRecord} where incident_category.deleted = 0 ${searchCondition} ${condition}`;
  let [incidentCategory] = await db.query(incidentCategoryQuery);
  incidentCategory = await decodeAndParseFields(incidentCategory);
     

  /**Count all organization */
  const totalRecord = await countQueryCondition(incidentCategoryQuery);

  return sendResponse(res, 200, incidentCategory, totalRecord);
};

/**Function to delete a specific Incident Category */
export const deleteIncidentCategory = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("incident_category", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "incident_category",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
