import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  settingsUpdater,
  deleteSettingRecord,
  encodeSingle_statement,
  decodeSingle_statement,
  uniqueIdGenerator,
} from "../helper/general.js";
import IncidentLocation from "../sequelize/IncidentLocationSchema.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";

/** Function to create or update IncidentLocation */
// export const createUpdateIncidentLocation = async (req, res) => {
//   const { id, name, description } = req.body;
//   const encodedDescription = await encodeSingle_statement(description);

//   if (id) {
//     const result = await settingsUpdater(
//       IncidentLocation,
//       req.body.organization,
//       req.body,
//       req.user
//     );
//     if (!result) {
//       return sendResponse(res, 404, "No data found");
//     }
//     await insertActivityLog(req.user.sessionid, "update", "IncidentLocation", `This user updated IncidentLocation with id: ${id}`);
//     return sendResponse(res, 200, "Record updated successfully");
//   } else {
//     const insertPromises = req.body.organization.map(async (org) => {
//       const [checkName] = await db.query(`SELECT name FROM incident_location WHERE name = ? AND deleted = 0 AND organization = ?`, [name, org]);
//       if (checkName.length === 0) {
//         await db.query(`INSERT INTO incident_location (name, description, created_by, organization) VALUES (?, ?, ?, ?)`, [name, encodedDescription, req.user.sessionid, org]);
//         await insertActivityLog(req.user.sessionid, "create", "IncidentLocation", `This user created a new IncidentLocation '${name}' for organization ${org}`);
//       }
//     });
//     await Promise.all(insertPromises);
//     return sendResponse(res, 200, "Record created successfully");
//   }
// };

export const createUpdateIncidentLocation = async (req, res) => {
  let { id, description, organization, department } = req.body;

  req.body.description = await encodeSingle_statement(description);

  req.body[req.body.id ? "updated_by" : "created_by"] = req.user.sessionid;

  /**If id comes in body then it will update the query */
  if (id) {
    const checkName = await db.query(
      `SELECT name FROM incident_location WHERE id = ? AND deleted = 0`,
      [id]
    );

    if (checkName.length == 0) {
      return sendResponse(res, 404, "Record not found");
    }

    if (checkName[0].name == "Over The Fence") {
      return sendResponse(res, 400, "Over The Fence cannot be updated");
    }
    if (organization.length === 0) {
      return sendResponse(
        res,
        404,
        "Please Select at least one business Structure"
      );
    }
    const result = await settingsUpdater(
      IncidentLocation,
      organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Incident Location",
      `This user Update Incident Location which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organization,
      department,
      "IL",
      "incident_location",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    const insertPromises = organization.map(async (element) => {
      // Check if the category type already exists for the organization
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM incident_location WHERE name = ? AND deleted = "0" AND organization = ?`,
        [req.body.name, element]
      );
      if (req.body.parent) {
        const [checkName] = await db.query(
          `SELECT name FROM incident_location WHERE id = ${req.body.parent}`
        );
        const [newParent] = await db.query(
          `SELECT id FROM incident_location WHERE name  = '${checkName[0].name}' AND organization = ${element}`
        );
        req.body.parent = newParent[0].id;
      }

      req.body.organization = element;
      if (checkNameWithOrganization.length === 0) {
        const { query, values } = createQueryBuilder(IncidentLocation, {
          ...req.body,
          organization: element,
        });
        // console.log(query, values);
        await db.query(query, values);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Incident Location",
          `This user created a new incident location '${req.body.name}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/** Function to view all or single IncidentLocation */
export const viewIncidentLocation = async (req, res) => {
  const { id } = req.params;
  const { dropdown } = req.query;

  let condition = await whereCondition({
    table: "incident_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    orderBy: req.query.orderBy,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "incident_location.name",
    "incident_location.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = incident_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = incident_location.organization",
    },
    {
      type: "left",
      targetTable: "incident_location AS parent_incident_location",
      onCondition: "parent_incident_location.id = incident_location.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  // const fetchQuery = `SELECT incident_location.*, CONCAT(users.name, ' ', users.surname) AS created_by, organization.name AS organization_name, FROM incident_location ${joinsRecord} WHERE incident_location.deleted = 0 ${searchCondition} ${condition}`;
  // let [records] = await db.query(fetchQuery);

  const [wherePart, orderByPart] = condition.split(/ORDER BY/i);
  condition = `
    ${wherePart.trim()}
    OR (incident_location.organization IS NULL AND incident_location.is_static = 1)
    ORDER BY ${orderByPart.trim()}
`;

  let fetchQuery = `SELECT incident_location.id, incident_location.unique_id, incident_location.created_by as created_by_id, incident_location.name,parent_incident_location.name AS parent_name , incident_location.description, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, organization.name as organization_name, incident_location.organization, incident_location.parent FROM incident_location ${joinsRecord} WHERE incident_location.deleted = 0 ${searchCondition} ${condition}`;
  // fetchQuery = fetchQuery.replace("ORDER BY incident_location.id DESC", "ORDER BY incident_location.id ASC");
  const [IncidentLocation] = await db.query(fetchQuery);

  async function fetchSubChildren(parentId) {
    const [subChildren] = await db.query(
      `SELECT * FROM incident_location WHERE deleted = 0 AND parent = ?`,
      [parentId]
    );

    // Only populate sub-children if any are found, and no need to recurse if not needed
    for (const child of subChildren) {
      child.description = await decodeSingle_statement(child.description);
      child.sub_children = await fetchSubChildren(child.id);
    }
    return subChildren;
  }

  // Map through IncidentLocation and only populate the parent along with its sub-children.
  const nestedIncidentLocations = [];

  for (const item of IncidentLocation) {
    item.description = await decodeSingle_statement(item.description);

    // Fetch the nested sub_children only for the top-level items.
    if (item.parent === null) {
      // Populate organizations for grouped data
      if (req.query.grouped == "true") {
        const [organizations] = await db.query(
          `SELECT organization FROM incident_location WHERE deleted = 0 AND name = ?`,
          [item.name]
        );
        const arr = organizations.map((item) => item.organization);
        item.organizations = arr;
      }
      // Fetch and add sub-children to the current illness type
      item.sub_children = await fetchSubChildren(item.id);
      nestedIncidentLocations.push(item); // Push only the parent with its nested structure
    } else if (dropdown != 1 || id) {
      nestedIncidentLocations.push(item);
    }
  }

  // for (const record of records) {
  //   record.description = await decodeSingle_statement(record.description);
  //   if (req.query.grouped == "true") {
  //     const [organizations] = await db.query(`SELECT organization FROM incident_location WHERE deleted = 0 AND name = ?`, [
  //       record.name,
  //     ]);
  //     const arr = organizations.map((item) => item.organization);
  //     record.organizations = arr;
  //   }
  // }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, nestedIncidentLocations, totalRecord);
};

export const getParentIncidentLocationList = async (req, res) => {
  let { filter } = req.query;
  filter = typeof filter === "string" ? JSON.parse(filter) : filter;
  let { organizations } = filter;
  if (!organizations || organizations.length == 0) {
    const [arrOrg] = await db.query(
      `SELECT id FROM organization WHERE deleted = 0`
    );
    organizations = arrOrg.map((item) => item.id);
    // return sendResponse(res, 400, "organizations is required");
  }
  // console.log(organizations);
  organizations =
    typeof organizations == "string"
      ? JSON.parse(organizations)
      : organizations;
  const parentIncidentLocationQuery = `SELECT r1.*
          FROM incident_location r1
          JOIN (
            SELECT name
            FROM incident_location
            WHERE organization IN (${organizations})
            GROUP BY name
            HAVING COUNT(DISTINCT organization) = ${organizations.length}
          )  r2 ON r1.name = r2.name
          WHERE r1.organization IN (${organizations}) AND r1.deleted = 0;
          `;
  // console.log(parentIncidentLocationQuery, "parentIncidentLocationQuery")
  const [parentIncidentLocationList] = await db.query(
    parentIncidentLocationQuery
  );
  const uniqueObj = {};
  parentIncidentLocationList.forEach((item) => {
    if (!uniqueObj[item.name]) {
      uniqueObj[item.name] = item;
    }
  });
  const uniqueArr = Object.values(uniqueObj);
  // console.log(uniqueArr);
  // return uniqueArr;

  return sendResponse(res, 200, uniqueArr);
};

/** Function to delete IncidentLocation */
export const deleteIncidentLocation = async (req, res) => {
  const { id } = req.params;
  const checkName = await db.query(
    `SELECT name FROM incident_location WHERE id = ? AND deleted = 0`,
    [id]
  );

  if (checkName.length == 0) {
    return sendResponse(res, 404, "Record not found");
  }

  if (checkName[0].name == "Over The Fence") {
    return sendResponse(res, 400, "Over The Fence cannot be deleted");
  }
  const deleteRecord = await deleteSettingRecord("incident_location", id);
  if (deleteRecord) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "IncidentLocation",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};

// /** Function to view all or single IncidentLocation */
// export const viewIncidentLocation = async (req, res) => {
//   const { id } = req.params;
//   const { dropdown } = req.query;

//   // Build condition for filtering, pagination, and grouping
//   const condition = await whereCondition({
//     table: "incident_location",
//     page: req.query.page,
//     all: req.query.all,
//     pageSize: req.query.pageSize,
//     filter: req.query.filter,
//     id,
//     user: req.user,
//     grouped: req.query.grouped,
//   });

//   // Search fields and condition
//   const searchFields = [
//     "incident_location.name",
//     "incident_location.description",
//     "users.name",
//     "organization.name",
//   ];
//   const searchCondition = await searchConditionRecord(
//     req.query.search,
//     searchFields
//   );

//   // Define joins
//   const joins = [
//     {
//       type: "left",
//       targetTable: "users",
//       onCondition: "users.id = incident_location.created_by",
//     },
//     {
//       type: "left",
//       targetTable: "organization",
//       onCondition: "organization.id = incident_location.organization",
//     },
//     {
//       type: "left",
//       targetTable: "incident_location AS parent_incident_location",
//       onCondition: "parent_incident_location.id = incident_location.parent",
//     },
//   ];
//   const joinsRecord = await makeJoins(joins);

//   // Base fetch query
//   const baseQuery = `
//   SELECT
//   incident_location.id,
//   incident_location.created_by AS created_by_id,
//   incident_location.name,
//       parent_incident_location.name AS parent_name,
//       incident_location.description,
//       users.name AS created_by,
//       users.surname AS created_by_surname,
//       users.profile AS created_by_profile,
//       organization.name AS organization_name,
//       incident_location.organization,
//       incident_location.parent
//     FROM
//       incident_location
//     ${joinsRecord}
//     WHERE
//       incident_location.deleted = 0
//       ${searchCondition}
//       ${condition}
//       `;
//   return console.log("baseQuery: ", baseQuery);

//   const [incidentLocations] = await db.query(baseQuery);

//   // Iterative function to fetch sub-children
//   async function fetchSubChildrenIterative(parentId) {
//     const stack = [parentId]; // Initialize stack with the parent ID
//     const subChildrenMap = {}; // To store sub-children for each parent

//     while (stack.length > 0) {
//       const currentParentId = stack.pop();

//       // Fetch direct children of the current parent
//       const [children] = await db.query(
//         `SELECT * FROM incident_location WHERE deleted = 0 AND parent = ?`,
//         [currentParentId]
//       );

//       subChildrenMap[currentParentId] = children;

//       // Process each child: decode description and add its ID to the stack
//       for (const child of children) {
//         child.description = await decodeSingle_statement(child.description);
//         stack.push(child.id);
//       }
//     }

//     // Transform the flat map into a nested structure
//     function buildNestedStructure(parentId) {
//       return (subChildrenMap[parentId] || []).map((child) => ({
//         ...child,
//         sub_children: buildNestedStructure(child.id),
//       }));
//     }

//     return buildNestedStructure(parentId);
//   }

//   // Build nested structure
//   const nestedIncidentLocations = [];

//   for (const item of incidentLocations) {
//     item.description = await decodeSingle_statement(item.description);

//     // Fetch sub-children for top-level items
//     if (item.parent === null) {
//       if (req.query.grouped === "true") {
//         const [organizations] = await db.query(
//           `SELECT organization FROM incident_location WHERE deleted = 0 AND name = ?`,
//           [item.name]
//         );
//         item.organizations = organizations.map((org) => org.organization);
//       }
//       item.sub_children = await fetchSubChildrenIterative(item.id); // Use iterative function
//       nestedIncidentLocations.push(item);
//     } else if (dropdown != 1 || id) {
//       nestedIncidentLocations.push(item);
//     }
//   }

//   // Calculate total records
//   const totalRecord = await countQueryCondition(baseQuery);

//   // Send the response
//   return sendResponse(res, 200, nestedIncidentLocations, totalRecord);
// };

export const viewIncidentLocationDropdown = async (req, res) => {
  const { id } = req.params;
  let condition = await whereCondition({
    table: "incident_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    orderBy: req.query.orderBy,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "incident_location.name",
    "incident_location.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = incident_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = incident_location.organization",
    },
    {
      type: "left",
      targetTable: "incident_location AS parent_incident_location",
      onCondition: "parent_incident_location.id = incident_location.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const [wherePart, orderByPart] = condition.split(/ORDER BY/i);
  condition = `
    ${wherePart.trim()}
    OR (incident_location.organization IS NULL AND incident_location.is_static = 1)
    ORDER BY ${orderByPart.trim()}
`;
  const fetchQuery = `SELECT incident_location.*, CONCAT(users.name, ' ', users.surname) AS created_by, organization.name AS organization_name,parent_incident_location.name AS parent_name FROM incident_location ${joinsRecord} WHERE incident_location.deleted = 0 ${searchCondition} ${condition}`;

  let [records] = await db.query(fetchQuery);

  // for (const record of records) {
  //   record.description = await decodeSingle_statement(record.description);
  //    if (req.query.grouped == "true") {
  //     const [organizations] = await db.query(
  //       `SELECT organization FROM incident_location WHERE deleted = 0 AND name = ?`,
  //       [record.name]
  //     );
  //     const arr = organizations.map((item) => item.organization);
  //     record.organizations = arr;
  //   }
  // }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};
