import {
  countQueryCondition,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import NearMissReporting from "../sequelize/NearMissReportingSchema.js";
import db from "../db-config.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createUpdateActionFunction,
  getCustomActionFunction,
} from "./customActionCreationController.js";

/**Function to create Safety Incident Form  */
export const createUpdateNearMissReporting = async (req, res) => {
  const { id, task_data, action_taken, sidebar_id } = req.body;
  req.body = (await decodeAndParseFields([req.body]))[0];
  const sessionId = req.user.sessionid;
  let organizationId = req.body.organization;
  const { department } = req.body;

  /** data for action creation */

  const data = {
    task_data,
    department,
    files: req.files,
    user: req.user,
    id: action_taken,
    organization: organizationId,
    sidebar_id,
  };

  if (req.body?.requires_investigation === "Yes") {
    const { status, arr } = await createUpdateActionFunction(data, req);
    if (status) {
      const action_taken = arr[0].id;
      req.body.action_taken = action_taken;
    }
  }

  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }
  req.body.notification = req.body.notification.map((user) => user.user_id);
  req.body.near_miss_submission = req.body.near_miss_submission.map(
    (user) => user.user_id
  );
  const unique_id = await uniqueIdGenerator(
    organizationId,
    department,
    "NEA",
    "near_miss_reporting",
    "unique_id",
    "unique_id"
  );
  const type = "reference";
  const reference_no = await uniqueIdGenerator(
    organizationId,
    department,
    "NEA",
    "near_miss_reporting",
    "reference_no",
    type
  );
  req.body.unique_id = unique_id;
  req.body.reference_no = reference_no;
  req.body = await encodeAndStringifyFields(req.body);
  req.body.created_by = sessionId;

  const { query, values } = id
    ? updateQueryBuilder(NearMissReporting, req.body)
    : createQueryBuilder(NearMissReporting, req.body);
  const [createNearMissForm] = await db.query(query, values);
  if (!id && createNearMissForm.insertId === 0) {
    return sendResponse(res, 404, "Cannot create record");
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Infrastructure Technical Form",
    `This user created a new Infrastructure Technical Form Record for organization ${organizationId}`
  );

  return sendResponse(
    res,
    200,
    `Record  ${id ? "updated" : "created"} successfully`
  );
};

export const getNearMissReporting = async (req, res) => {
  const { type } = req.query;
  const { id } = req.params;
  const condition = await whereCondition({
    table: "near_miss_reporting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    columnName: type,
  });
  const searchTableName = [
    "near_miss_reporting.near_miss_classification",
    "near_miss_reporting.immediate_action_taken",
    "near_miss_reporting.near_miss_description",
    "near_miss_reporting.severity",
    "near_miss_reporting.probability",
    "near_miss_reporting.investigation_outcome",
    "near_miss_reporting.immediate_cause",
    "near_miss_reporting.root_cause",
    "near_miss_reporting.near_miss_submission",
    "near_miss_reporting.reference_no",
    "near_miss_reporting.unique_id",
    "organization.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = near_miss_reporting.created_by",
    },
    {
      type: "left",
      targetTable: "users as reporter",
      onCondition: "reporter.id = near_miss_reporting.person_reporting",
    },
    {
      type: "left",
      targetTable: "department as reporter_department",
      onCondition: "reporter_department.id = reporter.department",
    },
    {
      type: "left",
      targetTable: "roles as reporter_role",
      onCondition: "reporter_role.id = reporter.role",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = near_miss_reporting.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = near_miss_reporting.department",
    },
  ];
  const joinCondition = await makeJoins(joins);
  const nearMissReportingQuery = `SELECT near_miss_reporting.* , CONCAT(reporter.name , ' ' , reporter.surname) as person_reporting_name , reporter.unique_id as person_reporting_employee_number , reporter.phone as reporter_contact , reporter_role.name as person_reporting_role_name ,reporter_department.name as person_reporting_department_name ,  organization.name as organization_name ,department.name as department_name ,CONCAT(users.name , ' ' , users.surname) as created_by FROM near_miss_reporting ${joinCondition} WHERE near_miss_reporting.deleted = 0 ${searchCondition} ${condition} `;
  const totalRecord = await countQueryCondition(nearMissReportingQuery);

  console.log('nearMissReportingQuery: ', nearMissReportingQuery);
  let [nearMissReporting] = await db.query(nearMissReportingQuery);
  nearMissReporting = await decodeAndParseFields(nearMissReporting);
  for (let i = 0; i < nearMissReporting.length; i++) {
    /** custom action creation data */

    if (
      nearMissReporting[i]?.action_taken &&
      nearMissReporting[i]?.requires_investigation === "Yes"
    ) {
      const data = {
        params: { id: nearMissReporting[i]?.action_taken },
        user: req.user,
        query: {},
      };
      const customActionData = await getCustomActionFunction(data);
      nearMissReporting[i].task_data = customActionData;
    }

    /** near miss classification */
    if (
      nearMissReporting[i].near_miss_classification &&
      nearMissReporting[i].near_miss_classification.length > 0
    ) {
      const [incidentCategoryList] = await db.query(
        `SELECT name from incident_category WHERE id In (${nearMissReporting[i].near_miss_classification}) AND deleted = 0`
      );

      nearMissReporting[i].near_miss_classification_name =
        incidentCategoryList.map((item) => item.name);
    }
    /** Near miss submission */
    const near_miss_submission = nearMissReporting[i].near_miss_submission;
    if (near_miss_submission && near_miss_submission.length > 0) {
      let [nearMissSubmission] = await db.query(
        `SELECT users.id as user_id, CONCAT(users.name , ' ' , users.surname) as name,  roles.name as role_name , department.name as department_name 
       FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE users.id IN (${near_miss_submission})`
      );
      nearMissReporting[i].near_miss_submission = nearMissSubmission;
    }
    /** Notification  User*/
    let [notification] = await db.query(
      `SELECT users.id as user_id, CONCAT(users.name , ' ' , users.surname) as name,  roles.name as role_name , department.name as department_name 
       FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE users.id IN (${nearMissReporting[i].notification})`
    );
    nearMissReporting[i].notification = notification;
  }
  return sendResponse(res, 200, nearMissReporting, totalRecord);
};

export const deleteNearMissReporting = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("near_miss_reporting", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Near Miss Reporting Form",
      `This user deleted a new near Miss Reporting Form Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};
