import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

import { sendResponse } from "../helper/wrapper.js";
import OhsFileRequirement from "../sequelize/OhsFileRequirementSchema.js";

export const createUpdateOhsFileRequirement = async (req, res) => {
  const { id, department } = req.body;
  req.user = {
    sessionid: 1,
  };

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0]?.organization;
  }

  let status = id ? "updated" : "created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  req.body = await encodeAndStringifyFields(req.body);

  if(!id){
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "OhsFileRequirement",
      "ohs_file_requirement",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(OhsFileRequirement, req.body)
    : createQueryBuilder(OhsFileRequirement, req.body);
  let [result] = await db.query(query, values);

  await insertActivityLog(
    req.user.sessionid,
    status,
    "OH&S File Requirement",
    id ? id : result.insertId
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getOhsFileRequirement = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "ohs_file_requirement",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "organization.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ohs_file_requirement.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = ohs_file_requirement.organization",
    },

    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = ohs_file_requirement.contractor",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const ohsFileRequirementQuery = `SELECT ohs_file_requirement.* , CONCAT(users.name , ' ' , users.surname ) AS created_by_name , organization.name AS organization_name , contractor_registration.contractor_name AS contractor_name FROM ohs_file_requirement ${joinCondition} WHERE ohs_file_requirement.deleted = 0 ${searchCondition} ${condition}`;

  let [ohsFileRequirement] = await db.query(ohsFileRequirementQuery);

  ohsFileRequirement = await decodeAndParseFields(ohsFileRequirement);

  const totalRecord = await countQueryCondition(ohsFileRequirementQuery);

  return sendResponse(res, 200, ohsFileRequirement, totalRecord);
};

export const deleteOhsFileRequirement = async (req, res) => {
  const { id } = req.params;

  await deleteRecord(OhsFileRequirement, id);

  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "OH&S File Requirement",
    id
  );

  return sendResponse(res, 200, "Record deleted successfully");
};
