import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
  insertNotification,
  uniqueIdGenerator,
  getPolicyComment,
  getRecord,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import PolicyComment from "../sequelize/PolicyCommentSchema.js";
import Policy from "../sequelize/PolicySchema.js";

/**Function to create/update Policy   */
export const policyCreateUpdate = async (req, res) => {
  const { id, organization, department, policy_reviewer, policy_status } =
    req.body;
  const policy_reviewer_list = [...policy_reviewer];
  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**If id comes in body then it will update the query */
  if (id) {
    /**Update Policy Query */
    /** First Check there status of policy is not complete or draft then only send notification to reviewer*/
    const [policyRecord] = await db.query(`SELECT * FROM policy WHERE id = ?`, [
      id,
    ]);

    /** Policy has been rejected then only amendment sop can only be created can't edit */
    if (policyRecord[0]?.current_status === "rejected") {
      return sendResponse(
        res,
        400,
        "Policy has been rejected Only Amendment SOP can be created"
      );
    }

    if (
      policyRecord[0].policy_status !== "complete" &&
      policy_status === "complete"
    ) {
      /** update the current_status */
      await db.query(
        "UPDATE policy SET current_status = 'reviewing' WHERE id = ?",
        [id]
      );
      const publicUrl = `/public-view-policy/${id}`;
      for (let i = 0; i < policy_reviewer_list.length; i++) {
        await insertNotification(
          "Policy Approval",
          publicUrl,
          policy_reviewer_list[i],
          "url",
          req.user.sessionid
        );
      }
    }
    req.body.updated_by = req.user.sessionid;
    const { query, values } = updateQueryBuilder(Policy, req.body);
    await db.query(query, values);
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "Policy", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const policyId = await uniqueIdGenerator(
      organizationId,
      department,
      "POL",
      "policy",
      "policy_id",
      "unique_id"
    );
    const type = "reference";
    const reference_no = await uniqueIdGenerator(
      organizationId,
      department,
      "POL",
      "policy",
      "reference_no",
      type
    );
    req.body.reference_no = reference_no;
    req.body.policy_id = policyId;
    req.body.created_by = req.user.sessionid;
    delete req.body.current_status;
    /**Insert record for Policy */
    const { query, values } = createQueryBuilder(Policy, req.body);
    const [createPolicy] = await db.query(query, values);
    if (policy_status === "complete") {
      /** update the current_status */
      await db.query(
        "UPDATE policy SET current_status = 'reviewing' WHERE id = ?",
        [createPolicy?.insertId]
      );
      /** First Check there status of policy is not complete or draft then only send notification to reviewer*/
      const publicUrl = `/public-view-policy/${createPolicy?.insertId}`;
      for (let i = 0; i < policy_reviewer_list.length; i++) {
        await insertNotification(
          "Policy Approval",
          publicUrl,
          policy_reviewer_list[i],
          "url",
          req.user.sessionid
        );
      }
    }
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Policy",
      createPolicy.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Policy */
export const viewPolicy = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "policy",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "policy.policy_title",
    "policy.policy_description",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];

  let policyComments = [];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `(p.policy_reviewer IN (${req.user.sessionid}) OR p.policy_owner = ${req.user.sessionid}) AND`;

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = policy.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = policy.policy_owner",
    },
    {
      type: "left",
      targetTable: "users as u3",
      onCondition: "u3.id = policy.policy_approver",
    },
    {
      type: "left",
      targetTable: "users as u4",
      onCondition: "u4.id = policy.policy_author",
    },
    {
      type: "left",
      targetTable: "document_classification",
      onCondition:
        "document_classification.id = policy.document_classification",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = policy.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = policy.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Policy */
  const PolicyQuery = `SELECT policy.*, CONCAT(u1.name , ' ' , u1.surname) AS created_by, CONCAT(u2.name , ' ' , u2.surname) AS policy_owner_name, CONCAT(u3.name , ' ' , u3.surname) AS policy_approver_name, CONCAT(u4.name , ' ' , u4.surname) AS policy_author_name, u1.profile AS created_by_profile, u2.profile AS policy_owner_profile, u3.profile AS policy_approver_profile, u4.profile AS policy_author_profile, document_classification.name AS document_classification_name, department.name AS department_name,
  organization.name AS organization_name, policy_reference,policy_revision_history,organization.header_image , organization.footer_image , organization.business_logo 
  FROM policy 
      ${joinsRecord} where policy.deleted = 0     AND policy.current_status IN ('pending', 'approved', 'rejected') 
 AND policy.created_by = ${req.user.sessionid}   ${searchCondition} ${condition}`;

  let [Policy] = await db.query(PolicyQuery);
  Policy = await decodeAndParseFields(Policy);
  for (let item of Policy) {
    delete item.updated_at;
    delete item.created_at;
    const policy_reviewer = item.policy_reviewer;
    if (policy_reviewer) {
      const [usersList] = await db.query(
        `SELECT id, CONCAT(name , ' ' , surname) as name, profile FROM users WHERE id IN (${policy_reviewer})`
      );
      item.policy_reviewer_details = usersList;
    }
    const roles_responsibilities = item?.roles_responsibilities || [];
    if (roles_responsibilities.length > 0) {
      for (let i = 0; i < roles_responsibilities.length; i++) {
        const role_id = roles_responsibilities[i]?.role_name;
        if (role_id) {
          const [record] = await getRecord("roles", "id", role_id);
          item.roles_responsibilities[i].name = record?.name;
        }
      }
    }
  }
  if (id) policyComments = await getPolicyComment(id);
  /**Count of all Policy */
  const totalRecord = await countQueryCondition(PolicyQuery);
  const dataWithComments = Policy.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.policy_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });
  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**Function to delete a specific Policy */
export const deletePolicy = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("policy", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Policy", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**Function to create comments on policy */
export const policyComment = async (req, res) => {
  const { status, policy_approver, policy_id } = req.body;
  const [record] = await getRecord("policy", "id", policy_id);
  // return console.log(record, "record"); || record?.current_status !== "approved"
  if (policy_approver) {
    const reviewerList = JSON.parse(record?.policy_reviewer || "[]");
    if (!reviewerList.includes(req.user.sessionid) && !req.user.isSuperAdmin) {
      return sendResponse(res, 200, "Access Denied");
    } else {
      await db.query(
        "Update policy set policy_approver = ? , current_status = 'approval' where id = ?",
        [policy_approver, req.body.policy_id]
      );
    }
  }
  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(PolicyComment, req.body);
  if (status == "rejected") {
    /** Update the policy status on the user basis if reviewer to draft or if approver then reject it*/
    const policy_reviewer = JSON.parse(record?.policy_reviewer || "[]");

    if (policy_reviewer.includes(req.user.sessionid)) {
      await db.query(
        "Update policy set policy_status = 'draft', current_status = 'rejected' where id = ?",
        [policy_id]
      );
    }
    if (
      record?.policy_approver == req.user.sessionid ||
      req.user.isSuperAdmin == 1
    ) {
      await db.query(
        "Update policy set policy_status = 'draft', current_status = 'rejected' where id = ?",
        [policy_id]
      );
    }
  }

  if (status == "approved") {
    /** if person is approver then update the policy status */
    if (
      record?.policy_approver == req.user.sessionid ||
      req.user.isSuperAdmin == 1
    ) {
      await db.query(
        "Update policy set  current_status = 'approved' where id = ?",
        [policy_id]
      );
    }
  }
  const insertedData = await db.query(query, values);
  return sendResponse(res, 200, "Record on policy stored");
};

/**Function to view all approval work flow and single Policy */
export const viewPolicyApprovalWorkFlow = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "policy",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  let policyComments = [];
  const searchTableName = [
    "policy.policy_title",
    "policy.policy_description",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `((JSON_CONTAINS(policy.policy_reviewer, '${req.user.sessionid}')) OR policy.policy_owner = ${req.user.sessionid} OR policy.policy_approver = ${req.user.sessionid}) AND`;

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = policy.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = policy.policy_owner",
    },
    {
      type: "left",
      targetTable: "users as u3",
      onCondition: "u3.id = policy.policy_approver",
    },
    {
      type: "left",
      targetTable: "users as u4",
      onCondition: "u4.id = policy.policy_author",
    },
    {
      type: "left",
      targetTable: "document_classification",
      onCondition:
        "document_classification.id = policy.document_classification",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = policy.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = policy.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Policy */
  const PolicyQuery = `SELECT policy.*, CONCAT(u1.name , ' ' , u1.surname) AS created_by, CONCAT(u2.name , ' ' , u2.surname) AS policy_owner_name, CONCAT(u3.name , ' ' , u3.surname) AS policy_approver_name, CONCAT(u4.name , ' ' , u4.surname) AS policy_author_name, u1.profile AS created_by_profile, u2.profile AS policy_owner_profile, u3.profile AS policy_approver_profile, u4.profile AS policy_author_profile, document_classification.name AS document_classification_name, department.name AS department_name,
  organization.name AS organization_name 
      FROM policy 
      ${joinsRecord} where policy.deleted = 0 AND ${approvalCondition} (policy.policy_status = 'complete')  ${searchCondition} ${condition}`;
  // return console.log(PolicyQuery, "PolicyQuery");
  let [Policy] = await db.query(PolicyQuery);
  Policy = await decodeAndParseFields(Policy);
  for (let item of Policy) {
    const policy_reviewer = item.policy_reviewer;
    item.public_url = `/public-view-policy/${item.id}`;
    if (policy_reviewer) {
      const [usersList] = await db.query(
        `SELECT id, CONCAT(name , ' ' , surname) as name, profile FROM users WHERE id IN (${policy_reviewer})`
      );
      item.policy_reviewer_details = usersList;
    }
  }

  if (id) policyComments = await getPolicyComment(id);
  // console.log(policyComments);
  /**Count of all Policy */
  const dataWithComments = Policy.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.policy_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });
  /**Count of all Policy */
  const totalRecord = await countQueryCondition(PolicyQuery);
  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**Function to view all approval work flow and single Policy */
export const viewPublicPolicy = async (req, res) => {
  const { id } = req.params;
  // get logged in user from session
  let loggedInUserId = req.user?.sessionid || 0;

  const condition = await whereCondition({
    table: "policy",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  let policyComments = [];
  const searchTableName = [
    "policy.policy_title",
    "policy.policy_description",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `AND ((JSON_CONTAINS(policy.policy_reviewer, '${req.user.sessionid}')) OR policy.policy_owner = ${req.user.sessionid} OR policy.policy_approver = ${req.user.sessionid})`;

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = policy.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = policy.policy_owner",
    },
    {
      type: "left",
      targetTable: "users as u3",
      onCondition: "u3.id = policy.policy_approver",
    },
    {
      type: "left",
      targetTable: "users as u4",
      onCondition: "u4.id = policy.policy_author",
    },
    {
      type: "left",
      targetTable: "document_classification",
      onCondition:
        "document_classification.id = policy.document_classification",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = policy.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = policy.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Policy */
  const PolicyQuery = `SELECT policy.*, CONCAT(u1.name , ' ' , u1.surname) AS created_by, CONCAT(u2.name , ' ' , u2.surname) AS policy_owner_name, CONCAT(u3.name , ' ' , u3.surname) AS policy_approver_name, CONCAT(u4.name , ' ' , u4.surname) AS policy_author_name, u1.profile AS created_by_profile, u2.profile AS policy_owner_profile, u3.profile AS policy_approver_profile, u4.profile AS policy_author_profile, document_classification.name AS document_classification_name, department.name AS department_name,
  organization.name AS organization_name 
      FROM policy 
      ${joinsRecord} where policy.deleted = 0  ${approvalCondition}  ${searchCondition} ${condition}`;
  let [Policy] = await db.query(PolicyQuery);
  Policy = await decodeAndParseFields(Policy);
  for (let item of Policy) {
    const policy_reviewer = item.policy_reviewer;
    item.public_url = `/public-view-policy/${item.id}`;
    if (policy_reviewer) {
      const [usersList] = await db.query(
        `SELECT id, CONCAT(name , ' ' , surname) as name, profile FROM users WHERE id IN (${policy_reviewer})`
      );
      item.policy_reviewer_details = usersList;
    }
  }

  if (id) policyComments = await getPolicyComment(id);
  // console.log(policyComments);
  /**Count of all Policy */
  const dataWithComments = Policy.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.policy_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });
  /**Count of all Policy */
  const totalRecord = await countQueryCondition(PolicyQuery);
  if (dataWithComments.length > 0) {
    // console.log('dataWithComments: ', dataWithComments);
    const reviewersArray = dataWithComments[0].policy_reviewer;
    if (
      Array.isArray(reviewersArray) &&
      reviewersArray.includes(loggedInUserId) &&
      dataWithComments[0].current_status === "reviewing"
    ) {
      // Use parameterized query to prevent SQL injection
      await db.query(
        `UPDATE policy SET policy_communicated_count = policy_communicated_count + 1 WHERE id = ?`,
        [dataWithComments[0].id]
      );
    }
  }

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**Function to change the policy status and update the approval work flow */
export const updatePolicyApprovalWorkflow = async (req, res) => {
  const { id } = req.params;
  const { policy_approver } = req.body;
  const [record] = await getRecord("policy", "id", id);
  const reviewerList = JSON.parse(record?.policy_reviewer || "[]");
  if (!reviewerList.includes(req.user.sessionid)) {
    return sendResponse(res, 200, "Access Denied");
  }
  const [query] = await db.query(
    `UPDATE policy SET policy_approver = '${policy_approver}', current_status = 'approval' WHERE id = ${id}`
  );
  return sendResponse(res, 200, "Record updated successfully");
};
