import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import RetentionPolicySetting from "../sequelize/RetentionPolicySettingSchema.js";

/**Function to create/update Retention Policy Setting   */
export const createUpdateRetentionPolicySetting = async (req, res) => {
  const { id, organization, department } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Update Retention Policy Setting Query */
    const { query, values } = updateQueryBuilder(
      RetentionPolicySetting,
      req.body
    );
    await db.query(query, values);
    /**Insert Activity  */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "RetentionPolicySetting",
      id
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    /**Insert record for Retention Policy Setting */
    const { query, values } = createQueryBuilder(
      RetentionPolicySetting,
      req.body
    );
    const createRetentionPolicySetting = await db.query(query, values);
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Retention Policy Setting",
      createRetentionPolicySetting.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Retention Policy Setting */
export const viewRetentionPolicySetting = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "retention_policy_setting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = retention_policy_setting.created_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = retention_policy_setting.department",
    },
    {
      type: "left",
      targetTable: "document_type",
      onCondition:
        "document_type.id = retention_policy_setting.document_type_id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = retention_policy_setting.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Retention Policy Setting */
  const RetentionPolicySettingQuery = `SELECT retention_policy_setting.*, document_type.name AS document_type_name, users.name AS created_by, 
  organization.name AS organization_name 
      FROM retention_policy_setting 
      ${joinsRecord} where retention_policy_setting.deleted = 0  ${searchCondition} ${condition}`;
  let [RetentionPolicySetting] = await db.query(RetentionPolicySettingQuery);

  RetentionPolicySetting = await decodeAndParseFields(RetentionPolicySetting);

  /**Count of all Retention Policy Setting */
  const totalRecord = await countQueryCondition(RetentionPolicySettingQuery);
  return sendResponse(res, 200, RetentionPolicySetting, totalRecord);
};

/**Function to delete a specific Retention Policy Setting */
export const deleteRetentionPolicySetting = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("retention_policy_setting", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "delete",
      "Retention Policy Setting",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};