import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  storeError,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
// import { getModules } from "../index.js";
import SoftwareAccessRequest from "../sequelize/SoftwareAccessRequestSchema.js";

export const createUpdateSoftwareAccessRequest = async (req, res) => {
  const { id } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);

  console.log("working", req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      req.body.department,
      "ContractorRegistration",
      "contractor_registration",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(SoftwareAccessRequest, req.body)
    : createQueryBuilder(SoftwareAccessRequest, req.body);
  const result = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Software Access Request",
    id ? id : result[0].insertId
  );

  return sendResponse(res, 200, `Record ${status} Successfully`);
};

/**Function to view all and single SoftwareAccessRequest */
export const viewSoftwareAccessRequest = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "software_access_request",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "software_access_request.reason",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "sidebar.title"
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = software_access_request.created_by",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = software_access_request.user_id",
    },
    {
      type: "left",
      targetTable: "sidebar",
      onCondition: "sidebar.id = software_access_request.sidebar_id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = software_access_request.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = software_access_request.department",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = u1.created_by",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const SoftwareAccessRequestQuery = `SELECT software_access_request.*, sidebar.title as sidebar_name, users.id AS created_by_id, u1.id AS user_id, u1.profile as user_profile, CONCAT(u1.name , ' ' , u1.surname) as user_name, users.name AS created_by, organization.name AS organization_name , users.surname as created_by_surname , users.profile as created_by_profile , department.name as department_name, CONCAT(u2.name , ' ' , u2.surname) as user_created_by, u2.id as user_created_by_id FROM software_access_request ${joinsRecord} where software_access_request.deleted = 0 and (u2.id = ${req.user.sessionid} or software_access_request.user_id = ${req.user.sessionid}) ${searchCondition} ${condition}`;

  let [SoftwareAccessRequestRecord] = await db.query(
    SoftwareAccessRequestQuery
  );

  SoftwareAccessRequestRecord = await decodeAndParseFields(
    SoftwareAccessRequestRecord
  );

  /**Count all organization */
  const totalRecord = await countQueryCondition(SoftwareAccessRequestQuery);

  return sendResponse(res, 200, SoftwareAccessRequestRecord, totalRecord);
};

/**Function to delete a specific SoftwareAccessRequest */
export const deleteSoftwareAccessRequest = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("software_access_request", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "SoftwareAccessRequest",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/** Function To Approve or reject SoftwareAccessRequest */
export const updateAccessRequestStatus = async (req, res) => {
  const { id, status } = req.body;

  let user_id = await getRecord("software_access_request", "id", id);
  let userId = user_id[0]?.user_id;
  let sidebar_id = user_id[0]?.sidebar_id;

  if (status === "approved") {
    const sidebarId = parseInt(sidebar_id);

    // Fetch user data
    let [userData] = await db.query(
      `SELECT organization, my_organization, permission FROM users WHERE id = ${userId}`
    );
    if (!userData.length) {
      throw new Error("User not found");
    }
    userData = userData[0];

    // Parse organization and permissions data
    const organizationArr = [
      userData.my_organization,
      ...JSON.parse(userData.organization),
    ];
    let permissionsData = JSON.parse(userData.permission);

    // Retrieve all relevant sidebar IDs
    const [sidebarChildren] = await db.query(
      `SELECT id FROM sidebar WHERE parent_id = ${sidebarId}`
    );
    const childrenIds = sidebarChildren.map((ele) => ele.id);
    const [subChildIds] = await db.query(
      `SELECT id FROM sidebar WHERE parent_id IN (${childrenIds})`
    );
    const allSidebarIds = [
      sidebarId,
      ...childrenIds,
      ...subChildIds.map((ele) => ele.id),
    ];

    // Set permissions for each sidebar
    const newPermissions = organizationArr.reduce((acc, orgId) => {
      acc[orgId] = { view: true, edit: true, delete: true, create: true };
      return acc;
    }, {});

    // Update or create permissions for each sidebarId
    allSidebarIds.forEach((sId) => {
      const sidebarEntry = permissionsData.find(
        (entry) => entry.sidebarId === sId
      );
      if (sidebarEntry) {
        sidebarEntry.permissions = {
          ...sidebarEntry.permissions,
          ...newPermissions,
        };
      } else {
        permissionsData.push({ sidebarId: sId, permissions: newPermissions });
      }
    });

    // Update permissions in the database
    await db.query(
      `UPDATE users SET permission = '${JSON.stringify(
        permissionsData
      )}' WHERE id = ${userId}`
    );
  }

  req.body.updated_by = req.user.sessionid;
  const { query, values } = updateQueryBuilder(SoftwareAccessRequest, {
    id,
    status,
  });
  await db.query(query, values);

  // Insert record for activity log
  await insertActivityLog(
    req.user.sessionid,
    "approved",
    "SoftwareAccessRequest",
    id
  );
  return sendResponse(res, 200, `Record approved Successfully`);
};

export const getPermission = async (req, res) => {
  try {
    const id = req.query.id || req.user.sessionid;
    const roleId = req.query.roleId;
    if (roleId) {
      let [permissionsData] = await db.query(
        `SELECT permission FROM roles WHERE id = ${roleId}`
      );
      permissionsData[0].permission = JSON.parse(permissionsData[0].permission);
      const modules = await getModules(permissionsData[0].permission || [], 0);
      return res.status(200).json({
        status: true,
        data: modules,
      });
    }
    let [permissionsData] = await db.query(
      `SELECT my_organization, permission FROM users WHERE id = ${id}`
    );
    permissionsData[0].permission = JSON.parse(permissionsData[0].permission);
    const modules = await getModules(
      permissionsData[0].permission || [],
      0,
      permissionsData[0].my_organization
    );
    return res.status(200).json({
      status: true,
      data: modules,
    });
  } catch (error) {
    storeError(error);
    res.status(500).json({
      status: false,
      message: "An error occurred",
      error: error.message,
    });
  }
};

export async function getModules(permissions, parentId, organization = null) {
  const [modules] = await db.query(
    `SELECT id, title, icon, path, type, dependent_module FROM sidebar WHERE parent_id = ${parentId} AND deleted = 0 ORDER BY sequence ASC`
  );

  for (const moduleData of modules) {
    const modulePermission = permissions.find(
      (permission) => permission.sidebarId === moduleData.id
    );

    if (modulePermission) {
      moduleData.permissions = organization
        ? { [organization]: modulePermission.permissions[organization] }
        : modulePermission.permissions;
    } else {
      moduleData.permissions = organization
        ? {
          [organization]: {
            view: false,
            edit: false,
            delete: false,
            create: false,
          },
        }
        : {};
    }

    moduleData.submodules = await getModules(
      permissions,
      moduleData.id,
      organization
    );
  }

  return modules;
}
