import db from "../db-config.js";
import StakeHolderIdentification from "../sequelize/StakeholderIdentificationSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

export const createUpdateStakeholderIdentification = async (req, res) => {
  const { id, department } = req.body;

  let status = id ? "Update" : "Create";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }

  if (req.body.external_stakeholder && req.body.external_stakeholder.length > 0) {
    const external_stakeholder = req.body.external_stakeholder;
    for (let i = 0; i < external_stakeholder.length; i++) {
      const stakeholder_individual_or_organization = external_stakeholder[i].stakeholder_individual_or_organization;
      if (stakeholder_individual_or_organization == 0) {
        req.body.external_stakeholder[i] = {
          ...req.body.external_stakeholder[i],
          organization_name: "",
          organization_contact_name: "",
          organization_contact_role: "",
          organization_contact_mobile: "",
          organization_telephone: "",
          organization_contact_email: "",
          organization_email: "",
          organization_address: "",
        };
      } else {
        req.body.external_stakeholder[i] = {
          ...req.body.external_stakeholder[i],
          individual_name: "",
          individual_email: "",
          individual_phone: "",
          individual_address: "",
        };
      }
    }
  }
  req.body = await encodeAndStringifyFields(req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "StakeHolderIdentification",
      "stakeholder_identification",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(StakeHolderIdentification, req.body)
    : createQueryBuilder(StakeHolderIdentification, req.body);
  // console.log(query, values);
  await db.query(query, values);
  await insertActivityLog(req.user.sessionid, status, "Stakeholder Identification", id);
  return sendResponse(res, 200, `Record ${status}d Successfully`);
};

export const getStakeholderIdentification = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "stakeholder_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });
  const searchTableName = [
    // "stakeholder_identification.classification",
    "stakeholder_identification.priority_level",
    "stakeholder_identification.classification_type",
    "stakeholder_classification.stakeholder_category",
    // "stakeholder_identification.is_individual_company",
    // "stakeholder_identification.individual_name",
    // "stakeholder_identification.individual_email",
    // "stakeholder_identification.individual_phone",
    // "stakeholder_identification.individual_address",
    // "stakeholder_identification.organization_name",
    // "stakeholder_identification.organization_contact_name",
    // "stakeholder_identification.organization_contact_role",
    // "stakeholder_identification.organization_contact_mobile",
    // "stakeholder_identification.organization_telephone",
    // "stakeholder_identification.organization_contact_email",
    // "stakeholder_identification.organization_email",
    // "stakeholder_identification.organization_address",
    // "stakeholder_identification.organization.name",
    // "stakeholder_identification.department.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = stakeholder_identification.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = stakeholder_identification.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = stakeholder_identification.department",
    },
    {
      type: "left",
      targetTable: " stakeholder_classification",
      onCondition: "stakeholder_classification.id = stakeholder_identification.stakeholder_category",
    },
  ];
  const joinRecord = await makeJoins(joins);

  const stakeholderIdentificationQuery = `SELECT stakeholder_identification.* , organization.name as organization_name , department.name AS department_name , users.name AS created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile ,stakeholder_classification.stakeholder_category AS stakeholder_category_name, stakeholder_classification.recommended_engagement_level ,stakeholder_classification.stakeholder_type AS stakeholder_type  FROM stakeholder_identification ${joinRecord} WHERE stakeholder_identification.deleted = 0 ${searchCondition} ${condition}`;

  let [stakeholderIdentification] = await db.query(stakeholderIdentificationQuery);
  stakeholderIdentification = await decodeAndParseFields(stakeholderIdentification);
  // console.log(stakeholderIdentification);
  for (let stakeholder of stakeholderIdentification) {
    const internal_stakeholders = stakeholder?.internal_stakeholder;

    if (Array.isArray(internal_stakeholders) && internal_stakeholders.length > 0) {
      const ids = internal_stakeholders
        .filter((e) => e.stakeholder_name_internal)
        .map((e) => e.stakeholder_name_internal);

      let usersData = [];
      if (ids.length > 0) {
        [usersData] = await db.query(
          `SELECT users.id, CONCAT(users.name, ' ', users.surname) AS name, users.profile, roles.name  AS role_name , department.name AS department_name
         FROM users 
         LEFT JOIN roles ON roles.id = users.role LEFT JOIN department ON department.id = users.department
         WHERE users.id IN (${ids.join(",")})`
        );
        const userMap = new Map(usersData.map((u) => [u.id, u]));
        // console.log("userMap", userMap);

        stakeholder.internal_stakeholder_details = internal_stakeholders.map((e) => {
          const user = userMap.get(e.stakeholder_name_internal) || {};
          // const departmentName =
          //   departmentMap.get(e.stakeholder_department_internal) ||
          //   "Unknown Department";

          return {
            name: user.name || "Unknown User",
            profile: user.profile || "Unknown Profile",
            role_name: user.role_name || "Unknown Role",
            id: user.id,
            department: user.department_name || "Unknown Department",
          };
        });
      }
    }
  }

  const totalRecord = await countQueryCondition(stakeholderIdentificationQuery);

  return sendResponse(res, 200, stakeholderIdentification, totalRecord);
};

export const deleteStakeholderIdentification = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(StakeHolderIdentification, id);
  await insertActivityLog(req.user.sessionid, "delete", "Stakeholder Identification", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

// export const getInternalStakeHolders = async (req, res) => {
//   const query = `SELECT internal_stakeholder FROM stakeholder_identification`;
//   const [internalStakeHolders] = await db.query(query);
//   const userIds = new Set();

//   internalStakeHolders.forEach((item) => {
//     const stakeholders = JSON.parse(item.internal_stakeholder || "[]");
//     stakeholders.forEach((stakeholder) => {
//       if (stakeholder.stakeholder_name_internal) {
//         userIds.add(stakeholder.stakeholder_name_internal);
//       }
//     });
//   });

//   // Convert Set to Array
//   const uniqueUserIds = Array.from(userIds);

//   if (uniqueUserIds.length === 0) {
//     return sendResponse(res, 200, [], "No internal stakeholder found");
//   }

//   const [usersData] = await db.query(
//     `SELECT users.id, CONCAT(users.name, ' ', users.surname) AS name, users.profile, roles.name  AS role_name , department.name AS department_name
//    FROM users
//    LEFT JOIN roles ON roles.id = users.role LEFT JOIN department ON department.id = users.department
//    WHERE users.id IN (${uniqueUserIds.join(",")})`
//   );
//   // const [rows] = await connection.execute(`SELECT id, name, department, role FROM users WHERE id IN (?)`, [
//   //   uniqueUserIds,
//   // ]);

//   return sendResponse(res, 200, usersData);
// };
export const getInternalStakeHolders = async (req, res) => {
  const query = `SELECT organization, internal_stakeholder FROM stakeholder_identification`;
  const [internalStakeHolders] = await db.query(query);

  // Use a Set to track unique combinations of organization and user ID
  const uniqueCombinations = new Set();
  const userMappings = [];

  internalStakeHolders.forEach((item) => {
    const stakeholders = JSON.parse(item.internal_stakeholder || "[]");
    stakeholders.forEach((stakeholder) => {
      if (stakeholder.stakeholder_name_internal) {
        const combinationKey = `${item.organization}-${stakeholder.stakeholder_name_internal}`;

        // Check if this combination already exists
        if (!uniqueCombinations.has(combinationKey)) {
          uniqueCombinations.add(combinationKey); // Add to the Set
          userMappings.push({
            userId: stakeholder.stakeholder_name_internal,
            organization: item.organization,
          });
        }
      }
    });
  });

  if (userMappings.length === 0) {
    return sendResponse(res, 200, [], "No internal stakeholder found");
  }

  // Extract user IDs for the database query
  const userIds = userMappings.map((mapping) => mapping.userId);

  // Fetch users data from the database
  const [usersData] = await db.query(
    `SELECT users.id, CONCAT(users.name, ' ', users.surname) AS name, users.email, users.profile, roles.name AS role_name, 
            department.name AS department_name
     FROM users 
     LEFT JOIN roles ON roles.id = users.role 
     LEFT JOIN department ON department.id = users.department
     WHERE users.id IN (${userIds.join(",")})`
  );

  // Include the organization for each user in the response
  const responseData = usersData.map((user) => {
    const userMapping = userMappings.find((mapping) => mapping.userId === user.id);
    return {
      ...user,
      organization: userMapping?.organization || null,
    };
  });

  return sendResponse(res, 200, responseData, "Internal stakeholders fetched successfully");
};
