import db from "../db-config.js";
import shortid from "shortid";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  insertActivityLog,
  insertNotification,
  insertSurveyQuestions,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import sendEmail from "../helper/sendEmail.js";

import { ApiResponse, sendResponse } from "../helper/wrapper.js";
import SurveyDesign from "../sequelize/SurveyDesignSchema.js";
import SurveyQuestions from "../sequelize/SurveyQuestionsSchema.js";
import SurveyResponse from "../sequelize/SurveyResponseSchema.js";
import ShortUrl from "../sequelize/ShortUrlSchema.js";

export const createUpdateSurveyDesign = async (req, res) => {
  const { id, questions } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  let totalCount = 0;
  if (questions) {
    for (const element of questions) {
      if (element.scoring_type == "automated") totalCount = totalCount + 1;
    }
    if (totalCount == questions.length) req.body.is_automated = 1;
  }
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(SurveyDesign, req.body)
    : createQueryBuilder(SurveyDesign, req.body);

  const [surveyDesign] = await db.query(query, values);

  const questionInserted = await insertSurveyQuestions(
    req.body.questions,
    id ? id : surveyDesign.insertId
  );
  if (!questionInserted)
    return sendResponse(res, 500, "Questions not inserted");

  const publicUrlInternal = `/survey/view/${id ? id : surveyDesign.insertId}`;
  await db.query(
    `UPDATE survey_design SET public_url = "${publicUrlInternal}" WHERE id = ${id ? id : surveyDesign.insertId
    }`
  );

  await insertActivityLog(
    req.user.sessionid,
    status === "Updated" ? "update" : "create",
    "Survey Design",
    `This user ${status} a new Survey Design Record`
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getSurveyDesign = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "survey_design",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "survey_design.name",
    "survey_design.description",
    "survey_design.survey_analysis",
    // "organization.name",
    // "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "survey_design.created_by = users.id",
    },
  ];
  const joinCondition = await makeJoins(joins);

  const surveyDesignQuery = `SELECT survey_design.*,users.name  as created_by_name,users.surname as created_by_surname , users.profile AS created_by_profile FROM survey_design ${joinCondition} WHERE survey_design.deleted = 0 ${searchCondition} ${condition} `;

  let [surveyDesign] = await db.query(surveyDesignQuery);

  surveyDesign = await decodeAndParseFields(surveyDesign);

  for (let survey of surveyDesign) {
    let [questions] = await db.query(
      `SELECT * FROM survey_questions WHERE survey_id = ${survey.id}`
    );
    questions = JSON.parse(JSON.stringify(questions));
    questions = await decodeAndParseFields(questions);
    survey.questions = questions;
    const [total] = await db.query(
      `SELECT COUNT(*) as total FROM survey_response WHERE survey_id = ${survey.id} AND response_values IS NOT NULL`
    );
    let totalRes = total[0]?.total;
    const [totalInternal] = await db.query(
      `SELECT COUNT(*) as total FROM survey_response WHERE survey_id = ${survey.id} AND response_values IS NOT NULL AND is_internal = 1`
    );
    let totalInternalRes = totalInternal[0]?.total;
    const [totalExternal] = await db.query(
      `SELECT COUNT(*) as total FROM survey_response WHERE survey_id = ${survey.id} AND response_values IS NOT NULL AND is_internal = 0`
    );
    let totalExternalRes = totalExternal[0]?.total;
    survey.total_survey_response = totalRes;
    survey.internal_response = totalInternalRes;
    survey.external_response = totalExternalRes;
  }

  const totalRecord = await countQueryCondition(surveyDesignQuery);
  return sendResponse(res, 200, surveyDesign, totalRecord);
};

export const deleteSurveyDesign = async (req, res) => {
  const { id } = req.params;

  await deleteRecord(SurveyDesign, id);
  await deleteRecord(SurveyQuestions, id, "survey_id");
  await deleteRecord(SurveyResponse, id, "survey_id");

  await insertActivityLog(req.user.sessionid, "delete", "Survey Design", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

export const enableDisableSurvey = async (req, res) => {
  // const { id } = req.params;
  const { status, id } = req.body;
  await db.query(
    `UPDATE survey_design SET status = ${status} WHERE id = ${id}`
  );
  await insertActivityLog(
    req.user.sessionid,
    "update",
    "Survey Design",
    `This user ${status} a new Survey Design Record`
  );
  return sendResponse(res, 200, "Record updated successfully");
};

export const sendOtpToExternalUser = async (req, res) => {
  let { email, name, survey_id } = req.body;

  const [surveyCheck] = await db.query(
    `SELECT status from survey_design WHERE id = ${survey_id}`
  );
  if (surveyCheck.length > 0 && surveyCheck[0].status == 0) {
    return sendResponse(res, 404, "Survey has been expired");
  }
  // email = email.trim();
  email = email ? email.replace(/\s+/g, "") : email;
  const [check] = await db.query(
    `SELECT * FROM survey_response WHERE email = '${email}' AND survey_id = ${survey_id} AND response_values IS NOT NULL`
  );
  if (check.length > 0) {
    return sendResponse(
      res,
      400,
      "User has already submitted response for this survey"
    );
  }

  const [check2] = await db.query(
    `SELECT * FROM survey_response WHERE email = '${email}' AND survey_id = ${survey_id} AND response_values IS NULL`
  );
  const id = check2.length > 0 ? check2[0].id : null;
  const [userCheck] = await db.query(
    `SELECT * , CONCAT(name , ' ' , surname) AS name from users WHERE email = '${email}' AND deleted = '0'`
  );
  req.body.is_internal = userCheck.length > 0 ? 1 : 0;
  req.body.user_id = userCheck.length > 0 ? userCheck[0].id : null;
  req.body.phone = userCheck.length > 0 ? userCheck[0].phone : null;
  req.body.name = userCheck.length > 0 ? userCheck[0].name : null;
  req.body.otp_expiry = Date.now() + 5 * 60 * 1000;
  if (id != null) {
    req.body.id = id;
  }
  const otp = Math.floor(100000 + Math.random() * 900000);
  req.body.otp = otp;

  const { query, values } = id
    ? updateQueryBuilder(SurveyResponse, req.body)
    : createQueryBuilder(SurveyResponse, req.body);
  const sendRecordArray = {
    templateFileUrl: "otpForSurveyTemplate.html",
    otp: otp,
    name: name,
    templateName: "Survey Design OTP",
  };
  const info = await sendEmail(
    "info@harmonyandhelp.com",
    email,
    "OTP verification for survey",
    sendRecordArray
  );
  await db.query(query, values);
  if (info) return sendResponse(res, 200, "Otp sent to email successfully");
  else return sendResponse(res, 400, "Otp sending failed. Please try again");
};

export const verifySurveyOtp = async (req, res) => {
  const { email, otp, survey_id } = req.body;
  const [check] = await db.query(
    `SELECT * FROM survey_response WHERE email = '${email}' AND survey_id = ${survey_id}`
  );
  if (check[0]?.otp_expiry < Date.now()) {
    await db.query(
      `UPDATE survey_response SET otp = null , otp_expiry = null WHERE id = ${check[0].id}`
    );
    return sendResponse(res, 400, "OTP expired. Please try again");
  }
  if (check[0]?.otp == otp) {
    await db.query(
      `UPDATE survey_response SET otp = null, otp_expiry = null WHERE id = ${check[0].id}`
    );
    return res.status(200).json({
      status: true,
      message: "OTP verified successfully",
      id: check[0].id,
    });
  } else
    return sendResponse(res, 400, "OTP verification failed. Please try again");
};

export const submitSurveyResponse = async (req, res) => {
  const { id, values } = req.body;
  const [check] = await db.query(
    `SELECT * FROM survey_response WHERE id = ${id}`
  );
  if (!values) {
    return sendResponse(res, 400, "Please fill all required fields");
  }
  if (check[0].values != null) {
    return sendResponse(
      res,
      400,
      "User has already submitted response for this survey"
    );
  }
  req.body.response_values = values;
  let totalScore = 0;
  for (const value of values) {
    let questionId = value.question_id;
    let answer = value.answer;
    let points = value.points;
    const [question] = await db.query(
      `SELECT * FROM survey_questions WHERE id = ${questionId}`
    );
    if (question[0]?.scoring_type == "automated") {
      if (
        question[0]?.answer_type == "Multiple Choice" ||
        question[0]?.answer_type == "Dropdown" ||
        question[0]?.answer_type == "Linear Scale"
      ) {
        const automatedScore = JSON.parse(question[0]?.automated_scores);

        let score = 0;
        score = answer != undefined ? Number(automatedScore[answer]) : 0;
        totalScore += score;
      } else if (question[0]?.answer_type == "Checkbox") {
        const automatedScore = JSON.parse(question[0]?.automated_scores);
        let score = 0;
        if (answer) {
          for (const ans of answer) {
            score = Number(automatedScore[ans]);
            totalScore += score;
          }
        }
      } else {
        if (answer != undefined) {
          totalScore += Number(question[0]?.points);
        }
      }
    } else {
      totalScore += points ? Number(points) : 0;
      if (points) req.body.is_declared = 1;
    }
  }
  req.body.total = totalScore;
  let [surveyDesign] = await db.query(
    `SELECT * FROM survey_design WHERE id = ${req.body.survey_id}`
  );
  surveyDesign = await decodeAndParseFields(surveyDesign);

  if (surveyDesign[0].is_automated) {
    req.body.is_declared = 1;
  }

  const surveyAnalysis = surveyDesign[0].survey_analysis;
  for (const element of surveyAnalysis) {
    const lower = element.starting;
    const upper = element.ending;
    if (totalScore >= lower && totalScore <= upper) {
      req.body.label = element.label;
      req.body.start_end = element.starting + "-" + element.ending;
      break;
    }
  }

  const { query, values: queryValues } = updateQueryBuilder(
    SurveyResponse,
    req.body
  );

  await db.query(query, queryValues);

  await db.query(
    `UPDATE survey_design SET is_responded = 1 WHERE id = ${req.body.survey_id}`
  );
  if (!check[0].response_values) {
    await insertNotification(
      "Filling the survey from",
      `${req.body.name} has successfully filled the survey form with id -> ${req.body.survey_id}`,
      1,
      "text",
      check[0].user_id ? check[0].user_id : req.body.name
    );
  }

  return sendResponse(res, 200, "Survey recorded successfully");
};

export const shareToInternal = async (req, res) => {
  const { id, user_id, internal_mail } = req.body;
  const [embedUrl] = await db.query(
    `SELECT public_url FROM survey_design WHERE id = ${id}`
  );
  for (const id of user_id) {
    await insertNotification(
      "Filling the survey from",
      embedUrl[0].public_url,
      id,
      "url",
      req.user.sessionid
    );
  }

  for (const item of internal_mail) {
    // Prepare the data to be inserted into the template
    const sendRecordArray = {
      templateFileUrl: "mail_for_survey_template.html",
      message: `Mail to Internal User`,
      name: item,
      link: `${process.env.HARMONY_FRONTEND_URL}${embedUrl[0].public_url}`,
      templateName: "Survey Design",
    };

    const subject = "Survey Design";

    // Send the email
    await sendEmail("info@harmonyandhelp.com", item, subject, sendRecordArray);
  }

  return res.status(200).json({
    status: true,
    message: `Survey sent successfully`,
  });
};

export const sendToMail = async (req, res) => {
  const { to, subject, message, url } = req.body;
  // Prepare the data to be inserted into the template
  const sendRecordArray = {
    templateFileUrl: "mail_for_survey_template.html",
    message: message,
    name: to,
    link: url,
    templateName: "Survey Design",
  };

  // Send the email
  await sendEmail("info@harmonyandhelp.com", to, subject, sendRecordArray);

  return sendResponse(res, 200, "Email sent successfully");
};

export const urlShorten = async (req, res) => {
  const { url } = req.body;

  if (!url) {
    return res.status(400).json({ error: "URL is required" });
  }

  const shortUrl = shortid.generate();

  req.body.shorten_url = shortUrl;
  req.body.original_url = url;
  const { query, values } = createQueryBuilder(ShortUrl, req.body);
  const fullShortUrl = `/${shortUrl}`;
  await db.query(query, values);
  return sendResponse(res, 200, fullShortUrl);
};

export const getShortUrl = async (req, res, next) => {
  const { shortUrl } = req.params;
  if (shortUrl.startsWith("socket.io")) return next();
  const [result] = await db.query(
    `SELECT * FROM short_url WHERE shorten_url = ?`,
    [shortUrl]
  );
  if (result.length == 0) return sendResponse(res, 404, "URL not found");
  let originalUrl = result[0]?.original_url;
  let clicks = Number(result[0]?.clicks) + 1;
  req.body.id = result[0]?.id;
  req.body.clicks = clicks;
  req.body.shorten_url = shortUrl;
  const { query, values } = updateQueryBuilder(ShortUrl, req.body);
  await db.query(query, values);
  if (!originalUrl) {
    return sendResponse(res, 404, "URL not found");
  }
  return res.redirect(originalUrl);
};
