import db from "../db-config.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import {
  uploadFile,
  createQueryBuilder,
  updateQueryBuilder,
  whereCondition,
  searchConditionRecord,
  makeJoins,
  decodeAndParseFields,
  countQueryCondition,
  deleteRecord,
  insertActivityLog,
  encodeAndStringifyFields,
  getRecord,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import TrainingCertificate from "../sequelize/TrainingCertificateSchema.js";

export const createUpdateTrainingCertificate = async (req, res) => {
  const { id, sidebar_id = 211 , organization, department } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body.attendees = JSON.parse(req.body.attendees);
  for (const i of req.body.attendees) {
    const id = i.user_id;
    const file = req?.files && req?.files[`upload_certificate_${id}`];
    console.log(file, "file");
    if (file) {
      // i.upload_certificate = await uploadFile("training_certificate", file);
      i.ddrm_id = await uploadToDDRM(sidebar_id, file, req);
    } else {
      i.upload_certificate = req.body[`upload_certificate_${id}`];
      // console.log("i.upload_certificate: ", i.upload_certificate);
      i.ddrm_id = i.ddrm_id || req.body[`ddrm_id_${id}`];
    }
  }

  const unique_id = await uniqueIdGenerator(
    organization,
    department,
    "TrainingCertificate",
    "training_certificate",
    "unique_id",
    "unique_id"
  );
  req.body.unique_id = unique_id;

  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(TrainingCertificate, req.body)
    : createQueryBuilder(TrainingCertificate, req.body);

  let [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Training Certificate",
    id ? id : result.insertId
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getTrainingCertificate = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "training_certificate",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "training_certificate.certificate_issued",
    "CONCAT(users.name , ' ' , users.surname)",
    "training_need_analysis.training_name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = training_certificate.created_by",
    },

    {
      type: "left",
      targetTable: "attendance_recording",
      onCondition:
        "attendance_recording.id = training_certificate.training_program",
    },
    {
      type: "left",
      targetTable: "training_schedule",
      onCondition:
        "training_schedule.id = attendance_recording.training_program",
    },
    {
      type: "left",
      targetTable: "training_need_analysis",
      onCondition:
        "training_need_analysis.id = training_schedule.training_program",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = training_need_analysis.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition:
        "department.id = training_schedule.internal_provider_department",
    },
  ];

  const joinCondition = await makeJoins(joins);

  let trainingCertificateQuery = `SELECT training_certificate.* , users.name  as created_by_name,users.surname AS created_by_surname , users.profile AS created_by_profile, training_need_analysis.training_name AS training_program_name , training_schedule.training_provider , COALESCE(training_schedule.single_day, training_schedule.multiple_day) AS training_date , organization.name AS organization_name FROM training_certificate  ${joinCondition} WHERE training_certificate.deleted = 0 ${searchCondition} ${condition} `;
  // console.log(trainingCertificateQuery)

  let [trainingCertificate] = await db.query(trainingCertificateQuery);

  trainingCertificate = await decodeAndParseFields(trainingCertificate);
  // console.log(trainingCertificate);
  for (const training of trainingCertificate) {
    const empId = training.attendees;
    for (const emp of empId) {
      const id = emp.user_id;
      const [user] = await db.query(
        `SELECT users.name , users.surname , users.profile , users.my_organization , users.unique_id AS employee_id , users.gender , users.disability , users.race , users.employee_type , users.role , roles.name AS role_name , organization.name AS organization_name FROM users LEFT join roles on roles.id = users.role LEFT JOIN organization ON organization.id = users.my_organization WHERE users.id = ?`,
        [id]
      );
      emp.name = user[0]?.name;
      emp.profile = user[0]?.profile;
      emp.organization = user[0]?.my_organization;
      emp.employee_id = user[0]?.employee_id;
      emp.gender = user[0]?.gender;
      emp.disability = user[0]?.disability;
      emp.race = user[0]?.race;
      emp.employee_type = user[0]?.employee_type;
      emp.role = user[0]?.role_name;
      emp.training_provider = training.training_provider;
      emp.training_date = training.training_date;
      emp.training_program_name = training.training_program_name;
      emp.organization_name = training.organization_name;
      // also add uploaded_certificate document file path from ddrm
      if (emp.ddrm_id) {
        const [certificateFile] = await getRecord(
          "repository",
          "id",
          emp.ddrm_id
        );
        if (certificateFile) {
          emp.upload_certificate = certificateFile.url;
        }
      }
    }
  }

  const totalRecord = await countQueryCondition(trainingCertificateQuery);
  return sendResponse(res, 200, trainingCertificate, totalRecord);
};

export const deleteTrainingCertificate = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(TrainingCertificate, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Training Certificate",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};

export const getTrainingCertificateEmployee = async (req, res) => {
  const { user_id } = req.params; // user id
  let { page = 1, pageSize = 10 } = req.query;
  page = parseInt(page, 10);
  pageSize = parseInt(pageSize, 10);
  let [trainingCertificate] = await db.query(`
    SELECT training_certificate.*, 
    training_need_analysis.training_name
    FROM training_certificate
    LEFT JOIN attendance_recording 
        ON attendance_recording.id = training_certificate.training_program
        LEFT JOIN training_schedule 
        ON training_schedule.id = attendance_recording.training_program
    LEFT JOIN training_need_analysis 
    ON training_need_analysis.id = training_schedule.training_program
    WHERE JSON_CONTAINS(
      training_certificate.attendees,
      JSON_OBJECT('user_id', ${user_id}),
      '$'
      ) 
      AND training_certificate.deleted = 0;
      `
    )
    
  trainingCertificate = await decodeAndParseFields(trainingCertificate);
  let userCertificates = [];
  let random_id = 1;
  for (const training of trainingCertificate) {
    const attendees = training.attendees;
    for (const attendee of attendees) {
      if (attendee.user_id == user_id) {
        const certificate = {
          id: random_id,
          user_id: attendee.user_id,
          name: attendee?.name + " " + attendee?.surname,
          profile: attendee?.profile,
          training_id: training.id,
          certificate_number: attendee?.certificate_number || "",
          certificate_expiry_date: attendee?.certificate_expiry_date || "",
          training_name: training.training_name,
        };
        userCertificates.push(certificate);
        random_id++;
      }
    }
  }
  const totalRecords = userCertificates.length;
  const totalPages = Math.ceil(totalRecords / pageSize);
  const currentPage = page;
  const startIndex = (currentPage - 1) * pageSize;
  userCertificates = userCertificates.slice(startIndex, startIndex + pageSize);
  return sendResponse(res, 200, userCertificates, totalRecords);
};
