import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  settingsUpdater,
  deleteSettingRecord,
  encodeSingle_statement,
  decodeSingle_statement,
  uniqueIdGenerator,
  encodeAndStringifyFields,
  updateQueryBuilder,
  createQueryBuilder,
  getFirstCreatedAndAllOrganizationIds,
  getFilterConditions,
  getListingData,
} from "../helper/general.js";
import TypeOfService from "../sequelize/TypeOfServiceSchema.js";

/** Function to create or update TypeOfService */
export const createUpdateTypeOfService = async (req, res) => {
  const { id, organization } = req.body;

  if (id && organization.length === 0) {
    return sendResponse(res, 404, "Please Select at least one business Structure");
  }

  if (!id) {
    const result = await getFirstCreatedAndAllOrganizationIds();

    if (!result.firstOrgId) {
      return sendResponse(res, 404, "No organization found.");
    }
    const unique_id = await uniqueIdGenerator(
      result.firstOrgId,
      req.body?.department || null,
      "TypeOfService",
      "type_of_service",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.organization = organization.length > 0 ? organization : result.organizationIds;
  }

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(TypeOfService, req.body)
    : createQueryBuilder(TypeOfService, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(req.user.sessionid, status, "TypeOfService", id ? id : result.insertId);
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

/** Function to view all or single TypeOfService */
export const viewTypeOfService = async (req, res) => {
  let { organizationCondition, condition } = await getFilterConditions(req, db, "type_of_service");

  const searchFields = ["type_of_service.name", "type_of_service.description", "users.name"];
  let searchCondition = await searchConditionRecord(req.query.search, searchFields);
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = type_of_service.created_by",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `
      SELECT type_of_service.*, 
             CONCAT(users.name, ' ', users.surname) AS created_by, 
             type_of_service.organization AS organizations 
      FROM type_of_service 
      ${joinsRecord} 
      WHERE type_of_service.deleted = 0 
      ${searchCondition} ${organizationCondition} ${condition}`;

  let [records] = await db.query(fetchQuery);

  records = await getListingData(records);

  // Create a lookup for child records based on parent_id
  const recordsById = Object.fromEntries(records.map((record) => [record.id, record]));
  records.forEach((record) => {
    record.sub_type = [];
    if (record.parent_id) {
      const parentRecord = recordsById[record.parent_id];
      if (parentRecord) {
        parentRecord.sub_type.push(record);
      }
    }
  });

  // Filter out only the root records (where parent_id is null)
  const rootRecords = records.filter((record) => !record.parent_id);

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, rootRecords, totalRecord);
};

/** Function to delete TypeOfService */
export const deleteTypeOfService = async (req, res) => {
  const { id } = req.params;
  const deleteRecord = await deleteSettingRecord("type_of_service", id);
  if (deleteRecord) {
    await insertActivityLog(req.user.sessionid, "delete", "TypeOfService", id);
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};

export const viewServiceType = async (req, res) => {
  const { id } = req.params;
  const { dropdown } = req.query;
  const condition = await whereCondition({
    table: "type_of_service",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "type_of_service.name",
    "type_of_service.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = type_of_service.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = type_of_service.organization",
    },
    {
      type: "left",
      targetTable: "type_of_service AS parent_type_of_service",
      onCondition: "parent_type_of_service.id = type_of_service.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const ServiceTypeQuery = `SELECT type_of_service.id, type_of_service.created_by as created_by_id, type_of_service.name,parent_type_of_service.name AS parent_name , type_of_service.description, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, organization.name as organization_name, type_of_service.organization, type_of_service.parent FROM type_of_service ${joinsRecord} WHERE type_of_service.deleted = 0 ${searchCondition} ${condition}`;
  const [ServiceType] = await db.query(ServiceTypeQuery);

  async function fetchSubChildren(parentId) {
    const [subChildren] = await db.query(`SELECT * FROM type_of_service WHERE deleted = 0 AND parent = ?`, [parentId]);

    // Only populate sub-children if any are found, and no need to recurse if not needed
    for (const child of subChildren) {
      child.description = await decodeSingle_statement(child.description);
      child.sub_children = await fetchSubChildren(child.id);
    }
    return subChildren;
  }

  // Map through ServiceType and only populate the parent along with its sub-children.
  const nestedServiceTypes = [];

  for (const item of ServiceType) {
    item.description = await decodeSingle_statement(item.description);

    // Fetch the nested sub_children only for the top-level items.
    if (item.parent === null) {
      // Populate organizations for grouped data
      if (req.query.grouped == "true") {
        const [organizations] = await db.query(
          `SELECT organization FROM type_of_service WHERE deleted = 0 AND name = ?`,
          [item.name]
        );
        const arr = organizations.map((item) => item.organization);
        item.organizations = arr;
      }

      // Fetch and add sub-children to the current Service type
      item.sub_children = await fetchSubChildren(item.id);
      nestedServiceTypes.push(item); // Push only the parent with its nested structure
    } else if (dropdown != 1 || id) {
      nestedServiceTypes.push(item);
    }
  }

  const totalRecord = await countQueryCondition(ServiceTypeQuery);
  return sendResponse(res, 200, nestedServiceTypes, totalRecord);
};
