import db from "../db-config.js";
import Excel from "exceljs";
import fs from "fs"
import path from "path";
import {
  decodeSingle_statement,
  makeJoins,
  storeError,
  textExtractor,
} from "./general.js";
import asyncHandler from "express-async-handler";
import { getCustomActionFunction } from "../controller/customActionCreationController.js";

export const objectiveAndTargetExcel = asyncHandler(async (req, res) => {
  try {
    const { id } = req.params; // objective settings id

    const joins = [
      {
        type: "left",
        targetTable: "users as objective_owner",
        onCondition: "objective_owner.id = objective_setting.objective_owner",
      },
      {
        type: "left",
        targetTable: "measurement_metric",
        onCondition:
          "measurement_metric.id = objective_setting.measurement_metric",
      },
    ];

    const joinRecord = await makeJoins(joins);

    const objectiveSettingQuery = `SELECT 
    objective_setting.id,
    objective_setting.objective_name as objective_title,
    objective_setting.date_generated,
    objective_setting.objective_description,
    objective_setting.human_resource_requirements as human_resources,
    objective_setting.technological_resources,
    objective_setting.numeric_value as numerical_value,
    objective_setting.type_of_resources,
    objective_setting.amount,
    objective_setting.target_name,
    objective_setting.kpi_name,
    objective_setting.target_description,
    objective_setting.management_plan_name as management_plan_title,
    objective_setting.management_plan_description,
    objective_setting.management_plan_target_id,
    objective_setting.organization,
    objective_setting.created_by as created_by_id,
    objective_setting.action_taken,
    measurement_metric.name as measurement_metric_name,
    CONCAT(objective_owner.name , ' ', objective_owner.surname) as objective_owner_name
    FROM objective_setting 
    ${joinRecord}
    WHERE objective_setting.id = ${id}`;
    const [objectiveSetting] = await db.query(objectiveSettingQuery);
    objectiveSetting[0].action_taken = JSON.parse(objectiveSetting[0].action_taken);
    for (let i = 0; i < objectiveSetting.length; i++) {
      const actionTaken = objectiveSetting[i].action_taken;
      if (actionTaken) {
        for (let record of actionTaken) {
          const data = {
            params: { id: record },
            user: req.user,
            query: {},
          };
          const [customActionData] = await getCustomActionFunction(data);
          objectiveSetting[i].human_resources = customActionData?.human_resource_requirements;
          objectiveSetting[i].technological_resources = customActionData?.technological_resources;
        }
      }
      objectiveSetting[i].objective_description = await decodeSingle_statement(
        objectiveSetting[i].objective_description
      );
      objectiveSetting[i].objective_description = textExtractor(
        objectiveSetting[i].objective_description
      );

      objectiveSetting[i].target_description = await decodeSingle_statement(
        objectiveSetting[i].target_description
      );
      objectiveSetting[i].target_description = textExtractor(
        objectiveSetting[i].target_description
      );

      objectiveSetting[i].management_plan_description =
        await decodeSingle_statement(
          objectiveSetting[i].management_plan_description
        );
      objectiveSetting[i].management_plan_description = textExtractor(
        objectiveSetting[i].management_plan_description
      );
    }

    let filePath = "";

    if (objectiveSetting.length > 0) {
      /** objective setting management plan */

      const [managementFetchQuery] = await db.query(
        `SELECT objective_setting_target_plan.action_title as actionTitle, 
  objective_setting_target_plan.action_description as actionDescription , department.name as responsibleDepartment , users.name as responsiblePerson, 
  DATE_FORMAT(objective_setting_target_plan.start_date, '%Y-%m-%d') as startDate,
  DATE_FORMAT(objective_setting_target_plan.due_date, '%Y-%m-%d') as dueDate,
  objective_setting_target_plan.action_progress_description as actionProgressDescription , objective_setting_target_plan.action_progress_status as actionProgressStatus  FROM objective_setting_target_plan LEFT JOIN users ON objective_setting_target_plan.responsible_person = users.id 
  LEFT JOIN department ON objective_setting_target_plan.responsible_department = department.id
   WHERE objective_setting_target_plan.objective_setting_id = ${id}`
      );
      for (let item of managementFetchQuery) {
        item.actionDescription = await decodeSingle_statement(
          item.actionDescription
        );
        item.actionDescription = textExtractor(item.actionDescription);
      }
      // console.log("managementFetchQuery", managementFetchQuery);
      filePath = await ObjectiveExcelStructureGenerator(
        objectiveSetting[0],
        managementFetchQuery
      );
    }
    return res.status(200).json({
      status: true,
      message: "Excel file created successfully",
      data: filePath,
    });
  } catch (error) {
    storeError(error);
    return res.status(500).json({
      status: false,
      message: error.message,
    });
  }
});

const ObjectiveExcelStructureGenerator = async (data, managementFetchQuery) => {
  const {
    objective_title,
    objective_owner_name,
    date_generated,
    objective_description,
    human_resources,
    technological_resources,
    type_of_resources,
    amount,
    target_title,
    measurement_metric_name,
    numerical_value,
    kpi_name,
    target_description,
    management_plan_title,
    management_plan_description,
  } = data;
  const workbook = new Excel.Workbook();
  const worksheet = workbook.addWorksheet("Management");

  // Define colors
  const grayColor = "808080";
  const blueColor = "00008B";
  const whiteTextColor = { color: { argb: "FFFFFFFF" } };

  const headers = [
    { header: "No.", key: "id", width: 10 },
    { header: "Action Title", key: "actionTitle", width: 20 },
    { header: "Action Description", key: "actionDescription", width: 20 },
    {
      header: "Responsible Department",
      key: "responsibleDepartment",
      width: 20,
    },
    { header: "Responsible Person", key: "responsiblePerson", width: 20 },
    { header: "Start Date", key: "startDate", width: 20 },
    { header: "Due Date", key: "dueDate", width: 20 },
    {
      header: "Action Progress",
      key: "actionProgressDescription",
      width: 20,
    },
    {
      header: "Action Progress Status",
      key: "actionProgressStatus",
      width: 20,
    },
  ];

  worksheet.mergeCells(`A1:M1`);
  worksheet.getCell("A1").value = "Objectives and Targets";
  worksheet.getCell("A1").alignment = { horizontal: "center" }; // Center align the text
  worksheet.getCell("A1").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: blueColor },
  };
  worksheet.getCell("A1").font = {
    size: 16,
    bold: true,
    ...whiteTextColor,
  };
  worksheet.getRow(1).height = 20;
  worksheet.getRow(9).height = 20;
  worksheet.getRow(11).height = 20;
  worksheet.mergeCells(`A2:C2`);
  worksheet.getCell("A2").value = "parent Objective";
  worksheet.getCell("A2").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };

  worksheet.mergeCells(`D2:E2`);
  worksheet.getCell(`D2`).value = "";
  worksheet.getCell(`D2`).alignment = { horizontal: "center" };
  worksheet.getCell(`F2`).value = "Objective title";
  worksheet.getCell("F2").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`G2:I2`);
  worksheet.getCell(`G2`).value = objective_title;
  worksheet.mergeCells(`J2:L2`);
  worksheet.getCell(`J2`).value = "Objective Owner";
  worksheet.getCell("J2").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.getCell("M2").value = objective_owner_name;
  const columnL = worksheet.getColumn("L");
  columnL.width = 25;
  const columnM = worksheet.getColumn("M");
  columnM.width = 25;
  worksheet.mergeCells(`A3:C3`);
  worksheet.getCell(`A3`).value = "Date Generated";
  worksheet.getCell("A3").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`D3:E3`);
  worksheet.getCell(`D3`).value = date_generated;
  worksheet.getCell(`D3`).alignment = { horizontal: "center" };
  worksheet.getCell(`F3`).value = "Objective Description";
  worksheet.getCell("F3").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`G3:M3`);
  worksheet.getCell(`G3`).value = objective_description;
  worksheet.mergeCells(`A4:C6`);
  worksheet.getCell(`A4`).value = "Resources Requirements";
  worksheet.getCell("A4").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`D4:E4`);
  worksheet.getCell(`D4`).value = "Human Resources";
  worksheet.getCell("D4").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.getCell(`D5`).value = "Role";
  worksheet.getCell(`D6`).value = human_resources
    ?.map((resource) => resource.role_name)
    .join(", ");
  worksheet.getCell(`E5`).value = "No. of People";
  worksheet.getCell(`E6`).value = human_resources
    ?.map((resource) => resource.no_of_peoples)
    .join(", ");
  worksheet.getCell(`F4`).value = "Technological Resources";
  worksheet.getCell("F4").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`F5:F6`);
  worksheet.getCell(`F5`).value = (technological_resources || []).join(", ");
  worksheet.mergeCells(`G4:H4`);
  worksheet.getCell("G4").value = "Financial Resources";
  worksheet.getCell("G4").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.getCell(`G5`).value = "Type of Resources";
  worksheet.getCell(`G6`).value = type_of_resources;
  worksheet.getCell(`H5`).value = "Amount";
  worksheet.getCell(`H6`).value = parseInt(amount);

  worksheet.mergeCells(`I4:M6`);
  worksheet.mergeCells(`A7:C7`);
  worksheet.getCell("A7").value = "Target Title";
  worksheet.getCell("A7").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells(`D7:E7`);
  worksheet.getCell("D7").value = "Target Title Value";
  worksheet.getCell("F7").value = "Measurement Metric";
  worksheet.getCell("F7").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells("G7:H7");
  worksheet.getCell("G7").value = measurement_metric_name;
  worksheet.mergeCells(`I7:J7`);
  worksheet.getCell("I7").value = "Numerical Value";
  worksheet.getCell("I7").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.getCell("K7").value = numerical_value;
  worksheet.getCell("L7").value = "KPI name";
  worksheet.getCell("L7").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.getCell("M7").value = kpi_name;
  worksheet.mergeCells("A8:C8");
  worksheet.getCell("A8").value = "Target Description";
  worksheet.getCell("A8").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells("D8:M8");
  worksheet.getCell("D8").value = target_description;
  worksheet.mergeCells("A9:M9");
  worksheet.getCell("A9").alignment = { horizontal: "center" };
  worksheet.getCell("A9").value = "Objective and Target Management Plan";
  worksheet.getCell("A9").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: blueColor },
  };
  worksheet.getCell("A9").font = { size: 16, bold: true, ...whiteTextColor };
  worksheet.mergeCells("A10:C10");
  worksheet.getCell("A10").value = "Management Plan Title";
  worksheet.getCell("A10").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells("D10:E10");
  worksheet.getCell("D10").value = management_plan_title;
  worksheet.getCell("F10").value = "Management Plan Description";
  worksheet.getCell("F10").fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: grayColor },
  };
  worksheet.mergeCells("G10:M10");
  worksheet.getCell("G10").value = management_plan_description;

  // Merge cells and set background color
  worksheet.mergeCells("A11:M11");
  const cellA1 = worksheet.getCell("A11");
  cellA1.value = "Management Plan Actions";
  cellA1.alignment = { horizontal: "center" };
  cellA1.fill = {
    type: "pattern",
    pattern: "solid",
    fgColor: { argb: blueColor },
  };
  cellA1.font = { size: 16, bold: true, ...whiteTextColor }; // Increase font size and make it bold

  worksheet.mergeCells("B12:D12");
  worksheet.mergeCells("G12:H12");
  worksheet.mergeCells("J12:K12");

  // Set headers and background color
  worksheet.getCell("A12").value = "No.";
  worksheet.getCell("B12").value = "Action Title";
  worksheet.getCell("E12").value = "Action Description";
  worksheet.getCell("F12").value = "Responsible Department";
  worksheet.getCell("G12").value = "Responsible Person";
  worksheet.getCell("I12").value = "Start Date";
  worksheet.getCell("J12").value = "Due Date";
  worksheet.getCell("L12").value = "Action Progress";
  worksheet.getCell("M12").value = "Action Progress Status";

  worksheet.getColumn("D").width = 20;
  worksheet.getColumn("E").width = 20;
  worksheet.getColumn("F").width = 20;
  worksheet.getColumn("G").width = 20;
  worksheet.getColumn("H").width = 20;
  worksheet.getColumn("I").width = 15;
  worksheet.getColumn("J").width = 15;
  worksheet.getColumn("K").width = 15;

  managementFetchQuery.forEach((row, index) => {
    const rowIndex = index + 13; // Assuming row index starts from 1
    const rowObject = worksheet.getRow(rowIndex); // Get the row object
    rowObject.getCell(1).value = index + 1; // Set cell value for the first cell in the row
    let indexForHeader = 0;
    for (let i = 1; i < 14; i++) {
      // Start from 1 since Excel column index starts from 1
      if (i !== 3 && i !== 4 && i !== 8 && i !== 11) {
        rowObject.getCell(i).value = row[headers[indexForHeader].key];
        indexForHeader = indexForHeader + 1;
      }
    }
  });
  managementFetchQuery.forEach((row, index) => {
    const rowIndex = index + 13;
    const rowObject = worksheet.getRow(rowIndex);
    rowObject.getCell(1).value = index + 1;
  });

  managementFetchQuery.forEach((row, index) => {
    const rowIndex = index + 13;
    worksheet.mergeCells(`B${rowIndex}:"D${rowIndex}`);
    worksheet.mergeCells(`G${rowIndex}:H${rowIndex}`);
    worksheet.mergeCells(`J${rowIndex}:K${rowIndex}`);
  });

  // Fill headers with gray color and set text color to white

  const secondRow = worksheet.getRow(12);
  secondRow.eachCell((cell) => {
    cell.fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: grayColor },
    };
  });

  worksheet.eachRow({ includeEmpty: false }, function (row, rowNumber) {
    // Iterate over each cell in the row
    row.eachCell({ includeEmpty: false }, function (cell, colNumber) {
      // Set the border for each cell
      cell.border = {
        top: { style: "thin" }, // Thin top border
        left: { style: "thin" }, // Thin left border
        bottom: { style: "thin" }, // Thin bottom border
        right: { style: "thin" }, // Thin right border
        // You can adjust the style to 'thin' or 'thick' for darker or bolder borders
      };
      cell.alignment = {
        wrapText: true,
        horizontal: "center",
        vertical: "middle",
      };
    });
  });

  // Define the file path where you want to save the Excel file
  const cwd = process.cwd();
  const publicFolderPath = path.join(cwd, "public");
  const excelFolderPath = path.join(publicFolderPath, "excel");
  if (!fs.existsSync(excelFolderPath)) {
    fs.mkdirSync(excelFolderPath, { recursive: true });
  }

  const filePath = path.join(excelFolderPath, "Objective_And_Target.xlsx");

  // Save the workbook
  await workbook.xlsx.writeFile(filePath);
  return `excel/Objective_And_Target.xlsx`;
};
