import Excel from "exceljs";
import path from "path";
import db from "../db-config.js";
import {
  decodeSingle_statement,
  makeJoins,
  whereCondition,
  storeError,
  textExtractor,
  decodeTheEditorContent,
} from "./general.js";
import fs from "fs";
import asyncHandler from "express-async-handler";
import { sendResponse } from "./wrapper.js";

const controlEffectivenessObj = {
  np_1: "Not Practicable",
  n_0: "None",
  "ri_0.5": "Requires Improvement",
  s_1: "Satisfactory",
};

const inherentFrequencyObj = {
  7: "Continuous",
  6: "Daily",
  5: "Weekly",
  4: "Monthly",
  3: "Quarterly",
  2: "Yearly",
  1: "10 Yearly",
};

const inherentConsequenceObj = {
  1: "None",
  3: "Insignificant",
  7: "Noticeable",
  15: "ImOrganisationant",
  40: "Serious",
  100: "Disaster",
};

const inherentProbabilityObj = {
  5: "Expected",
  4: "Probable",
  3: "Unusual",
  2: "Remote",
  1: "Improbable",
};

const controlEffectivenessClassificationObj = {
  99: "Satisfactory",
  90: "Satisfactory",
  80: "Satisfactory",
  79: "Requires Improvement",
  70: "Requires Improvement",
  60: "Requires Improvement",
  50: "Requires Improvement",
  49: "Unsatisfactory",
  40: "Unsatisfactory",
  30: "Unsatisfactory",
  20: "Unsatisfactory",
  10: "Unsatisfactory",
  9: "None",
};

export const getOperationalRiskAssessmentRegisterExcel = asyncHandler(async (req, res) => {
  try {
    const { filter } = req.body;
    if (!filter) {
      return res.status(400).json({
        success: false,
        message: "Filter data is required",
      });
    }
    req.query.filter = filter;
    // const filterData = JSON.parse(filter);
    // console.log("filterData: ", filterData);
    // // console.log('filterData: ', filterData);

    // const { organization, department, risk_register_id } = filterData;
    // console.log('organization, department, risk_register_id: ', organization, department, risk_register_id);
    // if (!organization || !risk_register_id) {
    //   return res.status(400).json({
    //     success: false,
    //     message: "Organization and risk register id are required",
    //   });
    // }
    const workbook = new Excel.Workbook();
    const worksheet = workbook.addWorksheet("Operational Risk Assessment");
    const white = "FFFFFF";
    const black = "000000";
    const blue = "90CAF9";
    const darkBlue = "000080";
    const violet = "9400D3";
    const red = "FF0000";
    const yellow = "FFF59D";
    const gray = "808080";
    const lightGray = "D3D3D3";
    const green = "A5D6A7";

    const headers = [
      { header: "Risk ID", key: "riskId", width: 10 },
      { header: "Work Area", key: "workArea", width: 25 },
      { header: "Main Process", key: "mainProcess", width: 25 },
      { header: "Activity / Service", key: "activity", width: 35 },
      { header: "Activity Frequency", key: "activityFrequency", width: 35 },
      { header: "Hazard", key: "hazard", width: 25 },
      { header: "Category", key: "category", width: 25 },
      { header: "Risk", key: "risk", width: 25 },
      { header: "Impact", key: "impact", width: 25 },
      { header: "Frequency", key: "frequency", width: 25 },
      { header: "Probability", key: "probability", width: 25 },
      { header: "Consequence", key: "consequence", width: 25 },
      { header: "Inherit Risk Rating", key: "inheritRiskRating", width: 35 },
      { header: "Risk Ranking", key: "riskRanking", width: 35 },
      { header: "Control Description", key: "controlDescription", width: 35 },
      { header: "Substitution", key: "substitution", width: 25 },
      {
        header: "Engineering Controls",
        key: "engineeringControls",
        width: 35,
      },
      {
        header: "Administrative Control",
        key: "administrativeControl",
        width: 35,
      },
      { header: "Last Resort", key: "lastResort", width: 25 },
      {
        header: "Control Effectiveness",
        key: "controlEffectiveness",
        width: 35,
      },
      {
        header: "Residual Risk Rating",
        key: "residualRiskRating",
        width: 25,
      },

      // {
      //   header: "Control Effectiveness Classification",
      //   key: "controlEffectivenessClassification",
      //   width: 25,
      // },
      // { header: "Risk Ranking", key: "riskRanking", width: 25 },
      // {
      //   header: "Is there an Opportunity for this Risk",
      //   key: "opportunityForRisk",
      //   width: 35,
      // },
      // {
      //   header: "If YES State the Opportunity",
      //   key: "opportunityState",
      //   width: 35,
      // },
      // {
      //   header: "Is this a Risk a Priority (YES/NO)",
      //   key: "riskPriority",
      //   width: 35,
      // },
    ];

    headers.forEach((header) => {
      const columnIndex = headers.indexOf(header) + 1;
      worksheet.getColumn(columnIndex).width = 25;
    });

    // workbook.getColumn("A").width = 10

    // Strategic Risk Assessment Register Header
    worksheet.mergeCells("A1:X1");
    worksheet.getCell("A1").value = "OPERATIONAL RISK ASSESSMENT REGISTER";
    worksheet.getCell("A1").alignment = {
      vertical: "middle",
      horizontal: "center",
    };
    worksheet.getCell("A1").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("A1").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };

    // Risk Register Owner Heading
    worksheet.mergeCells("A2:C2");
    worksheet.getCell("A2").value = "RISK OWNER:";
    worksheet.getCell("A2").alignment = {
      vertical: "middle",
      horizontal: "center",
    };
    worksheet.getCell("A2").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("A2").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };

    // Risk Owner Name
    worksheet.mergeCells("D2:F2");
    // worksheet.getCell("D2").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    // };
    worksheet.getCell("D2").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("D2").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };

    // RISK FACILITATOR NAME Heading
    worksheet.mergeCells("G2:I2");
    worksheet.getCell("G2").value = "RISK FACILITATOR NAME:";
    worksheet.getCell("G2").alignment = {
      vertical: "middle",
      horizontal: "center",
    };
    worksheet.getCell("G2").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("G2").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };

    // RISK FACILITATOR NAME
    worksheet.mergeCells("J2:X2");
    // worksheet.getCell("J2").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    // };
    worksheet.getCell("J2").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };
    worksheet.getCell("J2").font = { name: "Tahoma", size: 16, bold: false };

    // Merge Risk ID rows
    worksheet.mergeCells("A3:A4");
    worksheet.getCell("A3").value = "Risk ID";
    worksheet.getCell("A3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("B3:B4");
    worksheet.getCell("B3").value = "Work Area";
    worksheet.getCell("B3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("C3:C4");
    worksheet.getCell("C3").value = "Main Process";
    worksheet.getCell("C3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("D3:D4");
    worksheet.getCell("D3").value = "Activity / Service";
    worksheet.getCell("D3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("E3:E4");
    worksheet.getCell("E3").value = "Activity Frequency";
    worksheet.getCell("E3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("F3:F4");
    worksheet.getCell("F3").value = "Hazard";
    worksheet.getCell("F3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("G3:G4");
    worksheet.getCell("G3").value = "Category";
    worksheet.getCell("G3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("H3:H4");
    worksheet.getCell("H3").value = "Risk";
    worksheet.getCell("H3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    worksheet.mergeCells("I3:I4");
    worksheet.getCell("I3").value = "Impact";
    worksheet.getCell("I3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    //! INHERENT RISK HEADER   ---------------------
    worksheet.mergeCells("J3:N3");
    worksheet.getCell("J3").value = "INHERENT RISK ASSESSMENT";
    worksheet.getColumn("J").width = 12;
    worksheet.getColumn("K").width = 12;
    worksheet.getColumn("L").width = 12;
    worksheet.getColumn("M").width = 12;
    worksheet.getColumn("N").width = 12;
    worksheet.getCell("J3").alignment = {
      vertical: "middle",
      horizontal: "center",
    };
    worksheet.getCell("J3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    // Frequency Header
    worksheet.getCell("J4").value = "Frequency";
    worksheet.getCell("J4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("J4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("J").width = 14;

    // Probability  Header
    worksheet.getCell("K4").value = "Probability ";
    worksheet.getCell("K4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("K4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("K").width = 14;

    // Consequence Header
    worksheet.getCell("L4").value = "Consequence";
    worksheet.getCell("L4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("L4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("L").width = 14;

    // Consequence Header
    worksheet.getCell("L4").value = "Consequence";
    worksheet.getCell("L4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("L4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("L").width = 14;

    // Inherent Risk Rating Header
    worksheet.getCell("M4").value = "Inherent Risk Rating";
    worksheet.getCell("M4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("M4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("M").width = 14;

    // Risk Ranking Header
    worksheet.getCell("N4").value = "Risk Ranking";
    worksheet.getCell("N4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("N4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
      // border: {
      //   right: { style: "thin", color: { argb: "000000" } },
      // },
    };
    worksheet.getCell("N4").border = {
      right: { style: "thin", color: { argb: "000000" } },
    };
    worksheet.getColumn("N").width = 14;

    //! INHERENT RISK HEADER END  ---------------------

    worksheet.mergeCells("O3:U3");
    worksheet.getCell("O4").value = "Control Description";
    worksheet.getCell("O4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };

    //! Residual Risk Assessment HEADER   ---------------------
    // worksheet.mergeCells("P3:W3");
    worksheet.getCell("O3").value = "Residual Risk Assessment";
    worksheet.getColumn("O").width = 12;
    worksheet.getColumn("P").width = 12;
    worksheet.getColumn("Q").width = 12;
    worksheet.getColumn("R").width = 12;
    worksheet.getColumn("S").width = 12;
    worksheet.getColumn("T").width = 12;
    worksheet.getColumn("U").width = 12;
    worksheet.getColumn("V").width = 12;
    worksheet.getColumn("W").width = 12;
    worksheet.getCell("O3").alignment = {
      vertical: "middle",
      horizontal: "center",
    };
    worksheet.getCell("O3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getCell("O4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };

    // Substitution Header
    worksheet.getCell("P4").value = "Substitution (30%)";
    worksheet.getCell("P4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("P4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("P").width = 14;

    // Engineering Controls  Header
    worksheet.getCell("Q4").value = "Engineering Controls (25%)";
    worksheet.getCell("Q4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("Q4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("Q").width = 14;

    // Consequence Header
    worksheet.getCell("R4").value = "Administrative Controls (20%)";
    worksheet.getCell("R4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("R4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("R").width = 14;

    // Last Resort Header
    worksheet.getCell("S4").value = "Last Resort (15%)";

    worksheet.getCell("S4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("S4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("S").width = 14;

    // Control Effectiveness Header
    worksheet.getCell("T4").value = "Control Effectiveness";
    worksheet.getCell("T4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("T4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("T").width = 14;
    worksheet.getCell("T4").border = {
      right: { style: "thin", color: { argb: "000000" } },
    };

    // Residual Risk Rating Header
    worksheet.getCell("U4").value = "Residual Risk Rating";
    worksheet.getCell("U4").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
    worksheet.getCell("U4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray },
    };
    worksheet.getColumn("U").width = 14;

    // Control effectiveness Classification Header
    // worksheet.getCell("V4").value = "Control effectiveness Classification";
    // worksheet.getCell("V4").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    //   wrapText: true,
    // };
    // worksheet.getCell("V4").fill = {
    //   type: "pattern",
    //   pattern: "solid",
    //   fgColor: { argb: lightGray },
    // };
    // worksheet.getColumn("V").width = 14;

    // Risk Ranking Header
    // worksheet.getCell("V4").value = "Risk Ranking";
    // worksheet.getCell("V4").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    //   wrapText: true,
    // };
    // worksheet.getCell("V4").fill = {
    //   type: "pattern",
    //   pattern: "solid",
    //   fgColor: { argb: lightGray },
    // };
    // worksheet.getColumn("V").width = 14;

    // ! Residual Risk Assessment HEADER END  ---------------------

    // worksheet.mergeCells("W3:W4");
    // worksheet.getCell("U3").value = "Is there an Opportunity for this Risk";
    // worksheet.getCell("U3").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    //   wrapText: true,
    // };
    // worksheet.getCell("U3").fill = {
    //   type: "pattern",
    //   pattern: "solid",
    //   fgColor: { argb: lightGray },
    // };

    // worksheet.getColumn("V").width = 14;
    // worksheet.mergeCells("V3:V4");
    // // worksheet.mergeCells("V3:W3");
    // worksheet.getCell("V3").value = "If YES State the Opportunity";
    // worksheet.getCell("V3").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    //   wrapText: true,
    // };
    // worksheet.getCell("V3").fill = {
    //   type: "pattern",
    //   pattern: "solid",
    //   fgColor: { argb: lightGray },
    // };

    // worksheet.mergeCells("W3:W4");
    // // worksheet.mergeCells("W3:X3");
    // worksheet.getCell("W3").value = "Is this a Risk a Priority (YES/NO)";
    // worksheet.getCell("W3").alignment = {
    //   vertical: "middle",
    //   horizontal: "center",
    //   wrapText: true,
    // };
    // worksheet.getCell("W3").fill = {
    //   type: "pattern",
    //   pattern: "solid",
    //   fgColor: { argb: lightGray },
    // };

    const startCol = "A".charCodeAt(0); // ASCII value for 'A'
    const endCol = "X".charCodeAt(0); // ASCII value for 'Z'
    const row = 3;

    for (let col = startCol; col <= endCol; col++) {
      const cell = worksheet.getCell(`${String.fromCharCode(col)}${row}`);
      cell.border = {
        top: { style: "thin", color: { argb: "000000" } },
        left: { style: "thin", color: { argb: "000000" } },
        bottom: { style: "thin", color: { argb: "000000" } },
        right: { style: "thin", color: { argb: "000000" } },
      };
    }

    // * Format Complete ---------

    const { id } = req.params;
    const condition = await whereCondition({
      table: "operational_risk_identification",
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      filter: req.query.filter,
      id,
      grouped: req.query.grouped,
      user: req.user,
    });

    // const [dataFetch] = await db.query(
    //   `SELECT id as riskId, frequency, probability, consequences, risk_rating as inherentRiskRating,
    //   risk_ranking as riskRanking, substitution, engineering_control as engineeringControl,
    //   administrative_control as administrativeControl, last_resort as lastResort,
    //   risk_ranking as riskRankings, opportunity_identification as opportunityIdentification,
    //   opportunity_description as opportunityDescription, priority_confirmation as priorityConfirmation
    //   FROM operational_risk_identification
    //   WHERE deleted = 0 ${condition}`
    // );

    const [dataFetch] = await db.query(
      `SELECT operational_risk_identification.*,
      operational_risk.main_process AS mainProcess,
      CONCAT(approver.name, ' ', approver.surname) AS riskRegisterApprover,
      CONCAT(facilitator.name, ' ', facilitator.surname) AS riskAssessmentFacilitator,
      operational_activities.name AS operationalActivitiesName,
      operational_activities.work_id,
      operational_activities.frequency AS activityFrequency,
       operational_work_area.name AS workAreaName
       FROM operational_risk_identification
        LEFT JOIN operational_risk
        ON operational_risk_identification.risk_register_id = operational_risk.id
        LEFT JOIN operational_activities
        ON operational_risk_identification.activity_id = operational_activities.id
        LEFT JOIN operational_work_area
        ON operational_activities.work_id = operational_work_area.id
        LEFT JOIN users AS approver
        ON approver.id = operational_risk.risk_register_approver
        LEFT JOIN users AS facilitator
        ON facilitator.id = operational_risk.risk_assessment_facilitator
        WHERE operational_risk_identification.deleted = 0 ${condition}`
    );

    if (dataFetch.length == 0) {
      return sendResponse(res, 400, "No data found");
    }
    worksheet.getCell("D2").value = dataFetch[0].riskRegisterApprover;
    worksheet.getCell("J2").value = dataFetch[0].riskAssessmentFacilitator;

    // dataFetch.forEach(async (row, index) => {
    //   const controlAssessments = JSON.parse(row["control_assessments"] || "[]");

    //   // let frequencyId = row["frequency"]
    //   let frequency = inherentFrequencyObj[row["frequency"]];
    //   let probability = inherentProbabilityObj[row["probability"]];
    //   let consequences = inherentConsequenceObj[row["consequences"]];
    //   // let substitution = controlEffectivenessObj[row["substitution"]];

    //   // let engineeringControl = controlEffectivenessObj[row["engineeringControl"]];
    //   // let administrativeControl = controlEffectivenessObj[row["administrativeControl"]];
    //   // let lastResort = controlEffectivenessObj[row["lastResort"]];
    //   let impact = decodeSingle_statement(row["impact"]);
    //   impact = textExtractor(impact || "");

    //   let opportunityIdentification = row["opportunity_identification"] === 1 ? "YES" : "NO";

    //   let priorityConfirmation = +row["priority_confirmation"] === 1 ? "YES" : "NO";

    //   let opportunityDescription = null;

    //   if (opportunityIdentification === "YES") {
    //     opportunityDescription = await decodeTheEditorContent(row["opportunity_description"], "description");
    //   }

    //   // console.log("row",row)
    //   const rowIndex = index + 5; // Start from row 5
    //   worksheet.getCell(rowIndex, 1).value = row["id"];
    //   worksheet.getCell(rowIndex, 2).value = row["workAreaName"];
    //   worksheet.getCell(rowIndex, 3).value = row["mainProcess"];
    //   worksheet.getCell(rowIndex, 4).value = row["operationalActivitiesName"];
    //   worksheet.getCell(rowIndex, 5).value = row["activityFrequency"];
    //   worksheet.getCell(rowIndex, 6).value = row["hazard"];
    //   worksheet.getCell(rowIndex, 7).value = row["category"];
    //   worksheet.getCell(rowIndex, 8).value = row["name"];
    //   worksheet.getCell(rowIndex, 9).value = impact;
    //   worksheet.getCell(rowIndex, 10).value = frequency;
    //   worksheet.getCell(rowIndex, 11).value = probability;
    //   worksheet.getCell(rowIndex, 12).value = consequences;
    //   worksheet.getCell(rowIndex, 13).value = row["risk_rating"];
    //   worksheet.getCell(rowIndex, 14).value = row["risk_ranking"];
    //   worksheet.getCell(rowIndex, 15).value = row["control_assessments"];
    //   worksheet.getCell(rowIndex, 16).value = substitution;
    //   worksheet.getCell(rowIndex, 17).value = engineeringControl;
    //   worksheet.getCell(rowIndex, 18).value = administrativeControl;
    //   worksheet.getCell(rowIndex, 19).value = lastResort;
    //   worksheet.getCell(rowIndex, 20).value = row["effectiveness"];
    //   worksheet.getCell(rowIndex, 21).value = row["residual_rating"];
    //   worksheet.getCell(rowIndex, 22).value = row["control_assessments"];
    //   worksheet.getCell(rowIndex, 23).value = row["risk_ranking"];
    //   worksheet.getCell(rowIndex, 24).value = opportunityIdentification;
    //   worksheet.getCell(rowIndex, 25).value = opportunityDescription;
    //   worksheet.getCell(rowIndex, 26).value = priorityConfirmation;
    // });

    // worksheet.addRow(headers.map((header) => header.header));

    let currentRowIndex = 5; // Start from row 5
    console.log("dataFetch: ", dataFetch);
    for (let row of dataFetch) {
      const controlAssessments = JSON.parse(row["control_assessments"] || "[]");
      let impact = textExtractor((await decodeSingle_statement(row["impact"])) || "");
      let hazard = textExtractor((await decodeSingle_statement(row["hazard"])) || "");
      let mainProcess = textExtractor((await decodeSingle_statement(row["mainProcess"])) || "");
      // Write the main row (parent row)
      worksheet.getCell(currentRowIndex, 1).value = row["id"];
      worksheet.getCell(currentRowIndex, 2).value = row["workAreaName"];
      worksheet.getCell(currentRowIndex, 3).value = mainProcess;
      worksheet.getCell(currentRowIndex, 4).value = row["operationalActivitiesName"];
      worksheet.getCell(currentRowIndex, 5).value = row["activityFrequency"];
      worksheet.getCell(currentRowIndex, 6).value = hazard;
      worksheet.getCell(currentRowIndex, 7).value = row["category"];
      worksheet.getCell(currentRowIndex, 8).value = row["name"];
      worksheet.getCell(currentRowIndex, 9).value = impact;
      worksheet.getCell(currentRowIndex, 10).value = inherentFrequencyObj[row["frequency"]];
      worksheet.getCell(currentRowIndex, 11).value = inherentProbabilityObj[row["probability"]];
      worksheet.getCell(currentRowIndex, 12).value = inherentConsequenceObj[row["consequences"]];
      worksheet.getCell(currentRowIndex, 13).value = row["risk_rating"];
      worksheet.getCell(currentRowIndex, 14).value = row["risk_ranking"];
      // worksheet.getCell(currentRowIndex, 24).value = row["opportunity_identification"] === 1 ? "YES" : "NO";
      // worksheet.getCell(currentRowIndex, 25).value =
      //   row["opportunity_identification"] === 1
      //     ? await decodeTheEditorContent(row["opportunity_description"], "description")
      //     : null;
      // worksheet.getCell(currentRowIndex, 26).value = +row["priority_confirmation"] === 1 ? "YES" : "NO";

      // currentRowIndex++; // Move to the next row for subrows

      // Add subrows for controlAssessments
      controlAssessments.forEach((assessment) => {
        worksheet.getCell(currentRowIndex, 15).value = assessment.description || "";
        worksheet.getCell(currentRowIndex, 16).value = controlEffectivenessObj[assessment.substitution] || "";
        worksheet.getCell(currentRowIndex, 17).value = controlEffectivenessObj[assessment.engineering_control] || "";
        worksheet.getCell(currentRowIndex, 18).value = controlEffectivenessObj[assessment.administrative_control] || "";
        worksheet.getCell(currentRowIndex, 19).value = controlEffectivenessObj[assessment.last_resort] || "";
        worksheet.getCell(currentRowIndex, 20).value = assessment.effectiveness || "";
        worksheet.getCell(currentRowIndex, 21).value = row["residual_rating"];
        // worksheet.getCell(currentRowIndex, 23).value = row["risk_ranking"];

        currentRowIndex++; // Move to the next row
      });
    }
    const cwd = process.cwd();

    const directoryPath = path.join(cwd, "public", "excel");
    const filePath = path.join(directoryPath, "Operational_Risk_Assessment_Register.xlsx");
    // Ensure the directory exists
    if (!fs.existsSync(directoryPath)) {
      fs.mkdirSync(directoryPath, { recursive: true });
    }

    // Save the workbook
    await workbook.xlsx.writeFile(filePath);

    return res.status(200).json({
      status: true,
      message: "Excel file created successfully",
      data: `${process.env.HARMONY_BACKEND_URL}/excel/Operational_Risk_Assessment_Register.xlsx`,
    });
  } catch (error) {
    storeError(error);
    return res.status(500).json({
      status: false,
      message: error.message,
    });
  }
});
