import Excel from "exceljs";
import path from "path";
import db from "../db-config.js";
import { decodeSingle_statement, makeJoins, whereCondition, storeError, textExtractor } from "./general.js";
import fs from "fs";
import asyncHandler from "express-async-handler";

const romanRankObj = {
  I: "red",
  II: "yellow",
  III: "blue",
  IV: "green",
  V: "grey",
};

const colorObj = {
  white: "FFFFFF",
  black: "000000",
  blue: "90CAF9",
  yellow: "FFF59D",
  darkBlue: "000080",
  violet: "9400D3",
  red: "FF0000",
  gray: "808080",
  lightGray: "EEEEEE",
  green: "A5D6A7",
};

export const getStrategicRiskAssessmentRegisterExcel = asyncHandler(async (req, res) => {
  try {
    const { filter } = req.body;
    if (!filter) {
      return res.status(400).json({
        success: false,
        message: "Filter data is required",
      });
    }
    const filterData = JSON.parse(filter);
    // console.log('filterData: ', filterData);

    const { organization, department, risk_register_id } = filterData;
    // console.log('organization, department, risk_register_id: ', organization, department, risk_register_id);
    if (!organization || !risk_register_id) {
      return res.status(400).json({
        success: false,
        message: "Organization and risk register id are required",
      });
    }

    const workbook = new Excel.Workbook();
    const worksheet = workbook.addWorksheet("Strategic Risk Assessment Regis");
    const white = "FFFFFF";
    const black = "000000";
    const blue = "90CAF9";
    const darkBlue = "000080";
    const violet = "9400D3";
    const red = "FF0000";
    const yellow = "FFF59D";
    const gray = "808080";
    const lightGray = "D3D3D3";
    const green = "A5D6A7";
    const headers = [
      { header: "Risk ID", key: "riskId", width: 20 },
      { header: "Risk Name", key: "riskName", width: 25 },
      { header: "Risk Description", key: "riskDescription", width: 25 },
      { header: "Impact", key: "impact", width: 25 },
      {
        header: "Causes",
        key: "causes",
        width: 25,
      },
      // { header: "Risk Owner", key: "riskOwner", width: 25 },
      {
        header: "Likelihood",
        key: "likelihood",
        width: 25,
      },
      { header: "Consequence", key: "consequence", width: 25 },
      { header: "Risk Rating", key: "riskRating", width: 25 },
      { header: "Risk Ranking", key: "riskRanking", width: 25 },
      { header: "Control Name", key: "controlName", width: 25 },
      {
        header: "Control Design Intent: /(enter) Preventative (P)/Corrective( C)/Reactive (R )",
        key: "controlDesignIntent",
        width: 25,
      },
      {
        header: "Control Effectiveness",
        key: "controlEffectiveness",
        width: 25,
      },
      { header: "Risk Rating", key: "controlResidualRiskRating", width: 25 },
      {
        header: "Risk Ranking",
        key: "controlResidualRiskRanking",
        width: 25,
      },

      { header: "Risk Rating", key: "residualRiskRating", width: 25 },
      {
        header: "Risk Ranking",
        key: "residualRiskRanking",
        width: 25,
      },
    ];

    headers.forEach((header) => {
      const columnIndex = headers.indexOf(header) + 1;
      // worksheet.getRow(columnIndex).height = 20;
      // worksheet.getRow(columnIndex).eachCell((cell) => {
      //   cell.alignment = { wrapText: true }; //  enable text wrapping
      // })
      worksheet.getColumn(columnIndex).width = 25;
    });

    // Risk ID column width
    worksheet.getColumn("A").width = 8;

    // COMPANY HEADER ROW HEIGHT
    const heading = worksheet.getRow(1);
    heading.height = 60;
    heading.eachCell((cell) => {
      cell.fill = {
        type: "pattern",
        pattern: "solid",
        fgColor: { argb: gray },
      };
      cell.font = {
        name: "Tahoma",
        size: 16,
      };
    });

    // Strategic Risk Assessment Register Header
    const headerRowSecond = worksheet.getRow(2);
    headerRowSecond.height = 22;
    const headerRowThird = worksheet.getRow(3);
    headerRowThird.height = 22;
    const headerRowFourth = worksheet.getRow(4);
    headerRowFourth.height = 22;

    // COMPANY HEADER
    worksheet.mergeCells("A1:P1");
    worksheet.getCell("A1").value = "Company Header";
    worksheet.getCell("A1").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("A1").font = { name: "Calibri", size: 11, bold: false };

    // Strategic Risk Assessment Register Header
    worksheet.mergeCells("A2:P2");
    worksheet.getCell("A2").value = "Strategic Risk Assessment Register";
    worksheet.getCell("A2").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("A2").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("A2").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Register Owner Heading
    worksheet.mergeCells("A3:C3");
    worksheet.getCell("A3").value = "Risk Register Owner";
    worksheet.getCell("A3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("A3").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("A3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Register Owner Name
    worksheet.mergeCells("D3:F3");
    worksheet.getCell("D3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("D3").font = { name: "Tahoma", size: 16, bold: false };

    // Risk Assessment Date Heading
    worksheet.mergeCells("G3:I3");
    worksheet.getCell("G3").value = "Risk Assessment Date";
    worksheet.getCell("G3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("G3").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("G3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Assessment Date
    worksheet.mergeCells("J3:L3");
    worksheet.getCell("J3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("J3").font = { name: "Tahoma", size: 16, bold: false };

    // Risk Approval Status Header
    worksheet.mergeCells("M3:N3");
    worksheet.getCell("M3").value = "Risk Register Status";
    worksheet.getCell("M3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("M3").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("M3").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Approval Status
    worksheet.mergeCells("O3:P3");
    worksheet.getCell("O3").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("O3").font = { name: "Tahoma", size: 16, bold: false };

    // Risk Facilitator Name Header
    worksheet.mergeCells("A4:C4");
    worksheet.getCell("A4").value = "Risk Facilitator Name";
    worksheet.getCell("A4").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("A4").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("A4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Facilitator Name
    worksheet.mergeCells("D4:F4");
    worksheet.getCell("D4").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("D4").font = { name: "Tahoma", size: 16, bold: false };

    // Risk Review Date Header
    worksheet.mergeCells("G4:I4");
    worksheet.getCell("G4").value = "Risk Review Date";
    worksheet.getCell("G4").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("G4").font = { name: "Tahoma", size: 16, bold: false };
    worksheet.getCell("G4").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Risk Review Date
    worksheet.mergeCells("J4:L4");
    worksheet.getCell("J4").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("J4").font = { name: "Tahoma", size: 16, bold: false };

    // Merge Risk ID rows
    worksheet.mergeCells("A5:A6");
    worksheet.getCell("A5").value = "Risk ID";
    worksheet.getCell("A5").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("A5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: blue },
    };

    // Merge Risk Name rows
    worksheet.mergeCells("B5:B6");
    worksheet.getCell("B5").value = "Risk Name";
    worksheet.getCell("B5").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("B5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: darkBlue },
    };

    // Merge Risk description Rows
    worksheet.mergeCells("C5:C6");
    worksheet.getCell("C5").value = "Risk Description";
    worksheet.getCell("C5").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("C5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: darkBlue },
    };

    // merge impact
    worksheet.mergeCells("D5:D6");
    worksheet.getCell("D5").value = "Impact";
    worksheet.getCell("D5").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("D5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: darkBlue },
    };

    // merge causes
    worksheet.mergeCells("E5:E6");
    worksheet.getCell("E5").value = "Causes";
    worksheet.getCell("E5").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("E5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: red },
    };

    // INHERENT RISK HEADER
    worksheet.mergeCells("F5:I5");
    worksheet.getCell("F5").value = "INHERENT RISK";
    worksheet.getColumn("F").width = 12;
    worksheet.getColumn("G").width = 12;
    worksheet.getColumn("H").width = 12;
    worksheet.getColumn("I").width = 12;
    worksheet.getCell("F5").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("F5").font = { name: "Tahoma", size: 14, bold: false };
    worksheet.getCell("F5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: "FF808080" }, // Gray color
    };

    // Likelihood Header
    worksheet.getCell("F6").value = "Likelihood";
    worksheet.getCell("F6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("F6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("F6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: violet },
    };
    worksheet.getColumn("F").width = 14;

    // Consequence Header
    worksheet.getCell("G6").value = "Consequence";
    worksheet.getCell("G6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("G6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("G6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: violet },
    };
    worksheet.getColumn("G").width = 14;

    // Risk Rating Header
    worksheet.getCell("H6").value = "Risk Rating";
    worksheet.getCell("H6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("H6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("H6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: violet },
    };
    worksheet.getColumn("H").width = 14;

    // Risk Ranking Header
    worksheet.getCell("I6").value = "Risk Ranking";
    worksheet.getCell("I6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("I6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("I6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: violet },
    };
    worksheet.getColumn("I").width = 14;

    // Control Name Header
    worksheet.mergeCells("J5:J6");
    worksheet.getCell("J5").value = "Control Name";
    worksheet.getCell("J5").alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell("J5").font = { name: "Tahoma", size: 11, bold: false };
    worksheet.getCell("J5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray }, // Gray color
    };
    worksheet.getColumn("J").width = 25;

    // Control Design Intent Header
    worksheet.mergeCells("K5:K6");
    worksheet.getCell("K5").value = "Control Design Intent: /(enter) Preventative (P)/Corrective( C)/Reactive (R )";
    worksheet.getCell("K5").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("K5").font = { name: "Tahoma", size: 11, bold: false };
    worksheet.getCell("K5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray }, // Gray color
    };
    worksheet.getColumn("K").width = 25;

    // Control Effectiveness
    worksheet.mergeCells("L5:L6");
    worksheet.getCell("L5").value = "Control Effectiveness";
    worksheet.getCell("L5").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("L5").font = { name: "Tahoma", size: 11, bold: false };
    worksheet.getCell("L5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray }, // Gray color
    };
    worksheet.getColumn("L").width = 25;

    // Control Risk Rating
    worksheet.mergeCells("M5:M6");
    worksheet.getCell("M5").value = "Risk Rating";
    worksheet.getCell("M5").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("M5").font = { name: "Tahoma", size: 11, bold: false };
    worksheet.getCell("M5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray }, // Gray color
    };
    worksheet.getColumn("M").width = 20;

    // Control Risk Ranking
    worksheet.mergeCells("N5:N6");
    worksheet.getCell("N5").value = "Risk Ranking";
    worksheet.getCell("N5").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("N5").font = { name: "Tahoma", size: 11, bold: false };
    worksheet.getCell("N5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: lightGray }, // Gray color
    };
    worksheet.getColumn("N").width = 20;

    // Residual Risk Ranking Header
    worksheet.mergeCells("O5:P5");
    worksheet.getCell("O5").value = "RESIDUAL RISK";
    worksheet.getCell("O5").alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
      color: { argb: black },
    };
    worksheet.getCell("O5").font = { name: "Tahoma", size: 14, bold: false };
    worksheet.getCell("O5").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: gray }, // Gray color
    };

    // Residual Risk Rating
    worksheet.getCell("O6").value = "Risk Rating";
    worksheet.getCell("O6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("O6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("O6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };
    worksheet.getColumn("O").width = 12;

    // Residual Risk Ranking
    worksheet.getCell("P6").value = "Risk Ranking";
    worksheet.getCell("P6").alignment = { vertical: "middle", horizontal: "center", wrapText: true };
    worksheet.getCell("P6").font = { name: "Tahoma", size: 11, bold: false, color: { argb: white } };
    worksheet.getCell("P6").fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: green }, // Gray color
    };
    worksheet.getColumn("P").width = 12;

    // Merge any other column cells
    worksheet.mergeCells("M4:P4");

    worksheet.addRow(headers.map((header) => header.header));

    // const { id } = req.params;
    const condition = await whereCondition({
      table: "strategic_risk",
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      filter: req.query.filter,
      id: risk_register_id,
      grouped: req.query.grouped,
      user: req.user,
    });
    /**Make Joins according to tables */

    const joins = [
      {
        type: "left",
        targetTable: "strategic_risk_identification",
        onCondition: "strategic_risk.id = strategic_risk_identification.risk_register_id",
      },
      {
        type: "left",
        targetTable: "users as RRO",
        onCondition: "RRO.id = strategic_risk.risk_register_approver",
      },
      {
        type: "left",
        targetTable: "users as RF",
        onCondition: "RF.id = strategic_risk.risk_assessment_facilitator",
      },
      {
        type: "left",
        targetTable: "organization",
        onCondition: "organization.id = strategic_risk.organization",
      },
    ];

    const joinRecord = await makeJoins(joins);

    const [dataFetch] = await db.query(
      `SELECT 
          CONCAT(RRO.name , ' ' , RRO.surname) as riskRegisterOwner,
          CONCAT(RF.name , ' ' , RF.surname) as riskAssessmentFacilitator,
          strategic_risk.id as strategicRiskId,
          organization.header_image as companyHeaderImage,
          organization.footer_image as companyFooterImage,
          organization.name as organizationName,
          strategic_risk.assessment_date as riskAssessmentDate,
          strategic_risk.review_date as riskReviewDate,
          strategic_risk.status as riskRegisterStatus,
          strategic_risk_identification.id as riskId, 
          strategic_risk_identification.risk_name as riskName, 
          strategic_risk_identification.risk_description as riskDescription , 
          strategic_risk_identification.impact_description as impact, 
          strategic_risk_identification.causes_description as causes, 
          strategic_risk_identification.inherent_likelihood as likelihood , 
          strategic_risk_identification.inherent_consequence as consequence , 
          strategic_risk_identification.inherent_risk_rating as riskRating , 
          strategic_risk_identification.inherent_risk_ranking as riskRanking , 
          strategic_risk_identification.residual_risk_rating as residualRiskRating , 
          strategic_risk_identification.residual_risk_ranking as residualRiskRanking,
          strategic_risk_identification.control_identification as controlIdentification
          FROM strategic_risk
          ${joinRecord}
      WHERE strategic_risk.deleted = 0 AND strategic_risk_identification.deleted = 0 ${condition}`
    );
    // console.log('condition: ', condition);

    if (dataFetch.length == 0) {
      return res.status(200).json({
        success: true,
        message: "No data found",
      });
    }

    /**Decode content */
    for (const item of dataFetch) {
      item.riskDescription = await decodeSingle_statement(item.riskDescription);

      item.riskDescription = textExtractor(item.riskDescription);
      // item.hazard_description = await decodeSingle_statement(
      //   item.hazard_description
      // );

      item.impact = await decodeSingle_statement(item.impact);

      item.impact = textExtractor(item.impact);

      item.causes = await decodeSingle_statement(item.causes);

      item.causes = textExtractor(item.causes);

      item.controlIdentification = JSON.parse(item.controlIdentification);
      // console.log("item.controlIdentification: ", item.controlIdentification)

      // item.controlDescription = await decodeSingle_statement(
      //   item.controlDescription
      // );

      // item.controlDescription = textExtractor(item.controlDescription);

      // item.category = await decodeSingle_statement(
      //   item.category
      // );

      // item.category = textExtractor(item.category);

      // item.opportunity_description = await decodeTheEditorContent(
      //   item.opportunity_description,
      //   "description"
      // );
      // console.log("item: ", item);
    }
    //  console.log(dataFetch);

    const keyOrder = [
      "riskId",
      "riskName",
      "riskDescription",
      "impact",
      "causes",
      "likelihood",
      "consequence",
      "riskRating",
      "riskRanking",
      "control_name",
      "control_design_intent",
      "control_effectiveness",
      "residual_risk_rating",
      "residual_risk_ranking",
      "residualRiskRating",
      "residualRiskRanking",
    ];

    const otherKeysOrder = [
      "riskRegisterOwner",
      "riskAssessmentFacilitator",
      "riskAssessmentDate",
      "riskReviewDate",
      "riskRegisterStatus",
    ];

    // const headerFooterOrder = ["companyHeaderImage", "companyFooterImage"];

    // const cellMapHeaderFooter = {
    //   companyHeaderImage: ["A1"],
    //   companyFooterImage: ["A9"],
    // };

    // Define the public path for company images
    const publicPath = "public/";

    const cellMapping = {
      riskRegisterOwner: ["D3"],
      riskAssessmentFacilitator: ["D4"],
      riskAssessmentDate: ["J3"],
      riskReviewDate: ["J4"],
      riskRegisterStatus: ["O3"],
    };

    let currentRowIndex = 7; // Starting row index

    let headerImagePath ;
    let footerImagePath;

    dataFetch.forEach((row) => {
      const controlIdentificationArray = row.controlIdentification || [];
      const baseData = { ...row }; // Base data without controls
      delete baseData.controlIdentification; // Remove controlIdentification for merging logic

      const rowCache = []; // Store rows for merging and writing
      const mergedRangeStart = currentRowIndex; // Keep track of the start row for merging

      // Write rows for controlIdentification
      controlIdentificationArray.forEach((control) => {
        const rowData = {
          ...baseData,
          control_name: control.control_name,
          control_design_intent: control.control_design_intent,
          control_effectiveness: control.control_effectiveness,
          residual_risk_rating: control.residual_risk_rating,
          residual_risk_ranking: control.residual_risk_ranking,
        };

        rowCache.push(rowData);

        keyOrder.forEach((key, colIndex) => {
          // console.log('key: ', key);
          const value = rowData[key] || ""; // Default to empty string
          // worksheet.getCell("D3").value = value;
          worksheet.getCell(currentRowIndex, colIndex + 1).value = value;
          worksheet.getCell(currentRowIndex, colIndex + 1).font = { name: "Tahoma" };
          worksheet.getRow(currentRowIndex).height = 20;
          worksheet.getCell(currentRowIndex, colIndex + 1).alignment = {
            vertical: "middle",
            horizontal: "left",
            wrapText: true,
          };

          if (key === "riskRanking" || key === "residualRiskRanking" || key === "residual_risk_ranking") {
            worksheet.getCell(currentRowIndex, colIndex + 1).fill = {
              type: "pattern",
              pattern: "solid",
              fgColor: { argb: colorObj[romanRankObj[value]] },
            };
          }
        });

        currentRowIndex++; // Move to the next row
      });

      otherKeysOrder.forEach((key) => {
        const value = baseData[key] || ""; // Default to empty string
        const cells = cellMapping[key]; // Get the corresponding cells for this key
        if (cells) {
          cells.forEach((cell) => {
            worksheet.getCell(cell).value = value;
          });
        }
      });

      // File paths for header and footer images
      headerImagePath = `${publicPath}${baseData.companyHeaderImage}`; // Replace with dynamic path
      footerImagePath = `${publicPath}${baseData.companyFooterImage}`; // Replace with dynamic path

      // Add the header image spanning cells A1:E3
      addImageToCellRange(headerImagePath, worksheet, "A1", "P1", "Company Header");


      // Handle merging of identical rows for specific keys
      const mergeColumns = [
        "riskId",
        "riskName",
        "riskDescription",
        "impact",
        "causes",
        "likelihood",
        "consequence",
        "riskRating",
        "riskRanking",
        "residualRiskRating",
        "residualRiskRanking",
      ];

      mergeColumns.forEach((key) => {
        const colIndex = keyOrder.indexOf(key) + 1;
        const isMergeable = rowCache.every((rowData, _, arr) => rowData[key] === arr[0][key]);

        if (isMergeable) {
          // Check if range is already merged
          const startCell = worksheet.getCell(mergedRangeStart, colIndex);
          const endCell = worksheet.getCell(currentRowIndex - 1, colIndex);

          if (!startCell.isMerged && !endCell.isMerged) {
            worksheet.mergeCells(mergedRangeStart, colIndex, currentRowIndex - 1, colIndex);
            worksheet.getCell(mergedRangeStart, colIndex).alignment = {
              vertical: "middle",
              horizontal: "center",
              wrapText: true,
            };
          }
        }
      });

      // Handle rows without controlIdentification
      if (controlIdentificationArray.length === 0) {
        keyOrder.forEach((key, colIndex) => {
          const value = row[key] || ""; // Default to empty string
          worksheet.getCell(currentRowIndex, colIndex + 1).value = value;
          worksheet.getCell(currentRowIndex, colIndex + 1).font = { name: "Tahoma" };

          if (key === "riskRanking" || key === "residualRiskRanking" || key === "residual_risk_ranking") {
            worksheet.getCell(currentRowIndex, colIndex + 1).fill = {
              type: "pattern",
              pattern: "solid",
              fgColor: { argb: colorObj[romanRankObj[value]] },
            };
          }
        });

        currentRowIndex++; // Increment row for rows without controlIdentification
      }
    });

    const cwd = process.cwd();

    worksheet.eachRow({ includeEmpty: false }, function (row, rowNumber) {
      // Iterate over each cell in the row
      row.eachCell({ includeEmpty: false }, function (cell, colNumber) {
        // Set the border for each cell
        cell.border = {
          top: { style: "medium" }, // Thin top border
          left: { style: "medium" }, // Thin left border
          bottom: { style: "medium" }, // Thin bottom border
          right: { style: "medium" }, // Thin right border
          // You can adjust the style to 'medium' or 'thick' for darker or bolder borders
        };
        cell.alignment = { vertical: "middle", horizontal: "center" };
      });
    });

    // applyCss(worksheet, 3, 1, 17, gray, black);

    // applyCss(worksheet, 5, 1, 1, blue, white);
    // applyCss(worksheet, 5, 2, 4, darkBlue, white);
    // applyCss(worksheet, 5, 5, 6, red, white);
    // applyCss(worksheet, 5, 7, 10, violet, white);
    // applyCss(worksheet, 5, 11, 15, lightGray, black);
    // applyCss(worksheet, 5, 16, 17, green, white);

    // applyCss(worksheet, 6, 1, 1, blue, white);

    const lastRow = worksheet.lastRow; // Get the last row object
    const lastRowNumber = lastRow ? lastRow.number : 0; // Get the row number (0 if no rows)
    // console.log('lastRowNumber: ', lastRowNumber);

    // Add footer data in the next row
    const footerRow = lastRowNumber + 1;

    // COMPANY FOOTER
    worksheet.mergeCells(`A${footerRow}:P${footerRow}`);
    worksheet.getCell(`A${footerRow}`).value = "Company Footer";
    worksheet.getCell(`A${footerRow}`).alignment = { vertical: "middle", horizontal: "center" };
    worksheet.getCell(`A${footerRow}`).font = { name: "Calibri", size: 11, bold: false };

    worksheet.getRow(footerRow).height = 60;

     // Add the footer image spanning cells A20:E23
    addImageToCellRange(footerImagePath, worksheet, `A${footerRow}`, `P${footerRow}`, "Company Footer");


    // Function to add an image to a specific cell range
    function addImageToCellRange(imagePath, worksheet, startCell, endCell, defaultText) {
      if (imagePath && fs.existsSync(imagePath)) {
        const extension = path.extname(imagePath).replace(".", ""); // Get file extension
        const imageId = workbook.addImage({
          filename: imagePath,
          extension: extension, // Dynamically set the format
        });

        // Get the row and column indexes for the start and end cells
        const startCellRef = worksheet.getCell(startCell);
        const endCellRef = worksheet.getCell(endCell);

        const startRow = startCellRef.row - 1; // 0-based index
        const startCol = startCellRef.col - 1; // 0-based index
        const endRow = endCellRef.row; // 0-based index
        const endCol = endCellRef.col; // 0-based index

        //clear the cell range
        worksheet.getCell(startCell).value = "";
        worksheet.getCell(startCell).alignment = { vertical: "middle", horizontal: "center" };
        worksheet.getCell(startCell).font = { name: "Calibri", size: 11, bold: false };

        // Insert the image spanning the cell range
        worksheet.addImage(imageId, {
          tl: { col: startCol, row: startRow }, // Top-left corner
          br: { col: endCol, row: endRow }, // Bottom-right corner
        });
      } else {
        // If image is null or missing, set the default text
        // worksheet.mergeCells(`${startCell}:${endCell}`);
        const cell = worksheet.getCell(startCell);
        cell.value = defaultText || "No Image Available"; // Default text if not provided
        cell.alignment = { vertical: "middle", horizontal: "center" };
        cell.font = { name: "Calibri", size: 11, bold: false };
      }
    }


    const directoryPath = path.join(cwd, "public", "excel");
    const filePath = path.join(directoryPath, "Strategic_Risk_Assessment_Register.xlsx");
    // Ensure the directory exists
    if (!fs.existsSync(directoryPath)) {
      fs.mkdirSync(directoryPath, { recursive: true });
    }

    // Save the workbook
    await workbook.xlsx.writeFile(filePath);

    return res.status(200).json({
      status: true,
      message: "Excel file created successfully",
      data: `${process.env.HARMONY_BACKEND_URL}/excel/Strategic_Risk_Assessment_Register.xlsx`,
    });
  } catch (error) {
    storeError(error);
    return res.status(500).json({
      status: false,
      message: error.message,
    });
  }
});

// function applyCss(worksheet, row, startCol, endCol, BgColor, textColor) {
//   for (let col = startCol; col <= endCol; col++) {
//     const cell = worksheet.getCell(row, col);
//     cell.fill = {
//       type: "pattern",
//       pattern: "solid",
//       fgColor: { argb: BgColor },
//     };
//     cell.font = {
//       name: "Tahoma",
//       color: { argb: textColor },
//     };
//     cell.alignment = {
//       vertical: "middle",
//       horizontal: "center",
//       // wrapText: true,
//     };
//   }
// }