import cron from "node-cron";
import moment from "moment";
import db from "../db-config.js";
import { scheduleBackup } from "../controller/documentCreationController.js";
import { ddrmExpiryDateApproaching, operationalRiskReviewDate, overdueAndPendingAction, processUploadDocuments, riskRegisterReviewDateAndApproval } from "../helper/general.js";
import { backupDatabase } from "../helper/general.js";
import sendEmail from "../helper/sendEmail.js";

const hseAppointmentQuery = `
  SELECT hse_appointment.id, hse_appointment.end_date, CONCAT(appointer_user.name , ' ' , appointer_user.surname) as appointer_name,
  CONCAT(appointed_user.name, ' ', appointed_user.surname) as appointed_name,
  appointer_user.email as appointer_email, 
  appointed_user.email as appointed_email, 
  appointment_type.name as name_of_appointment 
  FROM hse_appointment
  INNER JOIN users as appointer_user ON appointer_user.id = hse_appointment.appoints_user_id
  INNER JOIN users as appointed_user ON appointed_user.id = hse_appointment.appointed_user_id
  INNER JOIN appointment_type on appointment_type.id = hse_appointment.name_of_appointment
  WHERE hse_appointment.deleted = "0"
`;
const [appointments] = await db.query(hseAppointmentQuery);

const meetingActionQuery = `
  SELECT meeting.meeting_title as meeting_name, meeting_notes.actions as action_details, 
  CONCAT(meeting_chairperson.name, ' ', meeting_chairperson.surname) as meeting_chairperson,
  CONCAT(meeting_scriber.name, ' ', meeting_scriber.surname) as meeting_scriber
  FROM meeting
  INNER JOIN meeting_notes on meeting_notes.meeting_id = meeting.id
  INNER JOIN meeting_recording on meeting_recording.scheduled_meeting = meeting.id
  INNER JOIN users as meeting_chairperson on meeting_chairperson.id = meeting.meeting_owner
  INNER JOIN users as meeting_scriber on meeting_scriber.id = meeting.meeting_owner
  WHERE meeting.deleted = "0" AND meeting_notes.deleted = "0" AND meeting_recording.deleted = "0"
  `
const [meetings] = await db.query(meetingActionQuery);


// Define the cron job
const cronJob = () => {
  cron.schedule("0 0 * * *", async () => {
    console.log("Task running at midnight:", new Date().toLocaleString());
    // await processUploadDocuments();

    // Add your task logic here
    // scheduleBackup("monthly");
    // Example: Cleanup old data, send email reports, etc.

  });

  cron.schedule("0 0 * * *", async () => {
    console.log("Midnight cron job scheduled!");
    console.log("Starting database backup at midnight:", new Date().toLocaleString());
    await backupDatabase();

    console.log('Checking appointments at midnight:', new Date().toLocaleString());

    /**
     * HSE Appointment End Date  function for cron job 
    */
    await hseAppointmentEndDate();
    /**
     * Meeting Action End Date function for cron job
     */
    await meetingActionEndDate();

    // The existing risk register`s review date is approaching
    // await riskRegisterReviewDateAndApproval("mail_for_strategic_tactical_register_review_date_approaching_template.html","Strategic Risk/ Tactical Risk Register Review Date")

    // Operational Risk Review date is approaching
    // await operationalRiskReviewDate()


    // Document expiry date reminder
    // await ddrmExpiryDateApproaching();

    // overdue status if end_date is exceeded and pending if start_date is exceeded
    // await overdueAndPendingAction()

  });


  // ! TO test the cron jon 
  // cron.schedule("*/10 * * * * *", async () => {
  //   console.log('Cron executed in 10 seconds')

  //   // await hseAppointmentEndDate();
  //   await meetingActionEndDate();
  // })
};



/**
 * Cron job for HSE appointment end date approaching, 
 * Email sent based on HSE appointment End date
 */
async function hseAppointmentEndDate() {
  const today = moment().startOf('day'); // Normalize to midnight

  for (const appointment of appointments) {
    const endDate = moment(appointment.end_date).startOf('day'); // Normalize endDate
    const daysRemaining = endDate.diff(today, 'days'); // Difference in days

    const sendRecordArray = {
      templateFileUrl: "mail_for_hse_appointment_end_date_template.html",
      templateName: "HSE Appointment End Date",
      appointer_name: appointment?.appointer_name || "",
      appointed_employee_name: appointment?.appointed_name || "",
      HSE_Legal_Appointment_Name: appointment?.name_of_appointment || "",
      end_date: moment(appointment.end_date).isValid() ? moment(appointment.end_date).format('MMMM Do YYYY, h:mm A') : "", // Example: January 22, 2025 10:30 AM
      no_of_days: daysRemaining, // Set no_of_days to the calculated daysRemaining
      message: "HSE Appointment End Date",
      emit: function () { } // Function (won't be serialized)
    };

    // ! TO test the function job
    // await sendEmail(
    //   "info@harmonyandhelp.com",
    //   appointment?.appointer_email || "info@harmonyandhelp.com",
    //   `Upcoming Expiry: HSE Appointment - ${appointment.name_of_appointment}`,
    //   sendRecordArray
    // );

    // Check specific intervals: 90, 60, 30 days before
    if ([90, 60, 30].includes(daysRemaining)) {
      await sendEmail(
        "info@harmonyandhelp.com",
        appointment?.appointer_email || "info@harmonyandhelp.com",
        `Upcoming Expiry: HSE Appointment - ${appointment.name_of_appointment}`,
        sendRecordArray
      );
    }
    // After 30 days, send weekly reminders on Mondays
    else if (daysRemaining < 30 && daysRemaining >= 0 && today.day() === 1) { // 1 = Monday in Moment.js
      await sendEmail(
        "info@harmonyandhelp.com",
        appointment?.appointer_email || "info@harmonyandhelp.com",
        `Upcoming Expiry: HSE Appointment - ${appointment?.name_of_appointment || ""}`,
        sendRecordArray
      );
    }
  }
}

/**
 * Cron job for Meeting Action due date approaching
 */
async function meetingActionEndDate() {
  try {
    const today = moment().startOf('day'); // Normalize to midnight
    let sendRecordsArray = []; // Array to store all valid send records

    for (const meeting of meetings) {
      const actionData = meeting.action_details ? JSON.parse(meeting.action_details) : [];

      for (const item of actionData) {
        // Calculate daysRemaining based on item's due_date
        const endDate = moment(item.due_date).startOf('day');
        const daysRemaining = endDate.diff(today, 'days');

        // Check if within 3 days before due date (including due date)
        if (daysRemaining >= 0 && daysRemaining <= 3) {
          const actionDataQuery = `
            SELECT CONCAT(users.name , ' ' , users.surname) as action_owner, 
            users.email as action_owner_email 
            FROM users WHERE id = ?
          `;
          const [userData] = await db.query(actionDataQuery, [item.assigned_to]);

          // Check if we have user data and all required fields
          if (userData?.length > 0 &&
            item?.action_discussion_name &&
            item?.meeting_action_description &&
            item?.due_date
          ) {

            const sendRecord = {
              templateFileUrl: "mail_for_meeting_action_due_template.html",
              templateName: "Meeting Action Due Reminder",
              action_owner: userData[0].action_owner,
              action_name: item.action_discussion_name,
              action_discussion: item.meeting_action_description,
              due_date: moment(item.due_date).isValid() ?
                moment(item.due_date).format('MMMM Do YYYY, h:mm A') : "",
              meeting_name: meeting.meeting_name || "",
              meeting_chairperson: meeting.meeting_chairperson || "",
              meeting_scriber: meeting.meeting_scriber || "",
              no_of_days: daysRemaining,
              emit: function () { }
            };

            // Send email for actions due within 3 days
            await sendEmail(
              "info@harmonyandhelp.com", // Replace with your sender email
              userData[0].action_owner_email || "info@harmonyandhelp.com", // Assuming email field exists
              `Reminder: Upcoming Due Date for Assigned Action Items - ${item.action_discussion_name}`,
              sendRecord
            );

            sendRecordsArray.push(sendRecord);
            // console.log('sendRecord: ', sendRecord);
          }
        }
      }
    }

    return sendRecordsArray;

  } catch (error) {
    console.log("error in (meetingActionEndDate)", error);
    throw error;
  }
}
export default cronJob;
