// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateCategoryType,
  deleteCategoryType,
  viewCategoryType,
} from "../controller/categoryTypeController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { categoryTypeCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Category Type
 *   description: API operations for managing all category Types
 */
const categoryTypeRoutes = Router();

/**
 * @swagger
 * /api/categoryType/create-update-category-type:
 *   post:
 *     summary: Create or update a category Type
 *     tags: [Category Type]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: {
 *             id: "id",
 *             name: "name",
 *             description: "description",
 *             organization: []
 *           }
 */
categoryTypeRoutes.post(
  "/create-update-category-type",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      83,
      "category_type"
    )(req, res, next);
  },
  validateRequest(categoryTypeCreateUpdateValidationSchema),
  controllerWrapper(createUpdateCategoryType)
);

/**
 * @swagger
 * /api/categoryType/get-category-types:
 *   get:
 *     summary: Get a list of all category Types
 *     tags: [Category Type]
 */
categoryTypeRoutes.get(
  "/get-category-types",
  verifyToken,
  sideBarPermissionCheck("view", 83, "category_type"),
  controllerWrapper(viewCategoryType)
);

/**
 * @swagger
 * /api/categoryType/get-category-types/{id}:
 *   get:
 *     summary: Get a specific record of category Type
 *     tags: [Category Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the category type to retrieve
 *         schema:
 *           type: string
 */
categoryTypeRoutes.get(
  "/get-category-types/:id",
  verifyToken,
  sideBarPermissionCheck("view", 83, "category_type"),
  controllerWrapper(viewCategoryType)
);

/**
 * @swagger
 * /api/categoryType/delete-category-types/{id}:
 *   delete:
 *     summary: Delete a specific record of category Type
 *     tags: [Category Type]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the category type to delete
 *         schema:
 *           type: string
 */
categoryTypeRoutes.delete(
  "/delete-category-types/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 83, "category_type"),
  controllerWrapper(deleteCategoryType)
);

export default categoryTypeRoutes;
