// Import necessary modules
import { Router } from "express";

import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  addCommentToContractorMeetingApproval,
  viewAllContractorMeetingApproval,
  viewAllContractorMeetingApprovalById,
} from "../controller/contractorMeetingApprovalController.js";
import { controllerWrapper } from "../helper/wrapper.js";

const contractorMeetingApprovalRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Contractor Meeting Approval
 *   description: API operations for managing all Contractor Contractor Meeting Approval
 */

/**
 * @swagger
 * /api/contractorMeetingApproval/get-contractor-meeting-approval:
 *   get:
 *     summary: Get list of all Contractor Contractor Meeting Approval
 *     tags: [Contractor Meeting Approval]
 *     responses:
 *       200:
 *         description: A list of all contractor meeting approvals
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
contractorMeetingApprovalRoutes.get(
  "/get-contractor-meeting-approval",
  verifyToken,
  sideBarPermissionCheck("view", 288, "contractor_meeting_recording"),
  controllerWrapper(viewAllContractorMeetingApproval)
);

/**
 * @swagger
 * /api/contractorMeetingApproval/get-contractor-meeting-approvalById/{meetingId}:
 *   get:
 *     summary: Get a specific record of contractor Contractor Meeting Approval by meetingId
 *     tags: [Contractor Meeting Approval]
 *     parameters:
 *       - in: path
 *         name: meetingId
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: A specific contractor meeting approval record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
contractorMeetingApprovalRoutes.get(
  "/get-contractor-meeting-approvalById/:meetingId",
  verifyToken,
  sideBarPermissionCheck("view", 288, "contractor_meeting_recording"),
  controllerWrapper(viewAllContractorMeetingApprovalById)
);

/**
 * @swagger
 * /api/contractorMeetingApproval/add-comment-to-contractor-meeting-approval:
 *   post:
 *     summary: Add comment and approval status to a specific Contractor Contractor Meeting Approval
 *     tags:
 *       - Contractor Meeting Approval
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: number
 *               status:
 *                 type: string
 *               comment:
 *                 type: string
 *           example:
 *             id: 1
 *             status: "Approved"
 *             comment: "Everything looks good!"
 *     responses:
 *       200:
 *         description: Successful operation
 *       400:
 *         description: Bad request
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
contractorMeetingApprovalRoutes.post(
  "/add-comment-to-contractor-meeting-approval",
  verifyToken,
  sideBarPermissionCheck("edit", 288, "contractor_meeting_recording"),
  controllerWrapper(addCommentToContractorMeetingApproval)
);

export default contractorMeetingApprovalRoutes;
