import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateFinancialYear,
  viewFinancialYear,
  deleteFinancialYear,
} from "../controller/financialYearController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const financialYearRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Financial Year
 *   description: API operations for managing all Financial Years
 */

/**
 * @swagger
 * /api/financialYear/create-update-financial-year:
 *   post:
 *     summary: Create or update a financial year
 *     tags: [Financial Year]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *                 example: "FY 2024-2025"
 *               description:
 *                 type: string
 *                 example: "Financial Year for 2024-2025"
 *               department:
 *                 type: integer
 *                 example: 3
 *               organization:
 *                 type: integer
 *                 example: 1
 *             required:
 *               - name
 *               - description
 *     responses:
 *       200:
 *         description: Financial year created or updated successfully
 *       400:
 *         description: Bad request
 */
financialYearRoutes.post(
  "/create-update-financial-year",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      296,
      "financial_year"
    )(req, res, next);
  },
  controllerWrapper(createUpdateFinancialYear)
);

/**
 * @swagger
 * /api/financialYear/get-financial-year:
 *   get:
 *     summary: Get all financial years
 *     tags: [Financial Year]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of financial years
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                     example: 1
 *                   name:
 *                     type: string
 *                     example: "FY 2024-2025"
 *                   description:
 *                     type: string
 *                     example: "Financial Year for 2024-2025"
 *                   department:
 *                     type: integer
 *                     example: 3
 *                   organization:
 *                     type: integer
 *                     example: 1
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-08-01T00:00:00.000Z"
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-08-15T00:00:00.000Z"
 *       400:
 *         description: Bad request
 */
financialYearRoutes.get(
  "/get-financial-year",
  verifyToken,
  sideBarPermissionCheck("view", 296, "financial_year"),
  controllerWrapper(viewFinancialYear)
);

/**
 * @swagger
 * /api/financialYear/get-financial-year/{id}:
 *   get:
 *     summary: Get a financial year by ID
 *     tags: [Financial Year]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the financial year
 *     responses:
 *       200:
 *         description: A financial year
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   example: 1
 *                 name:
 *                   type: string
 *                   example: "FY 2024-2025"
 *                 description:
 *                   type: string
 *                   example: "Financial Year for 2024-2025"
 *                 department:
 *                   type: integer
 *                   example: 3
 *                 organization:
 *                   type: integer
 *                   example: 1
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-08-01T00:00:00.000Z"
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-08-15T00:00:00.000Z"
 *       400:
 *         description: Bad request
 */
financialYearRoutes.get(
  "/get-financial-year/:id",
  verifyToken,
  sideBarPermissionCheck("view", 296, "financial_year"),
  controllerWrapper(viewFinancialYear)
);

/**
 * @swagger
 * /api/financialYear/delete-financial-year/{id}:
 *   delete:
 *     summary: Delete a financial year by ID
 *     tags: [Financial Year]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the financial year
 *     responses:
 *       200:
 *         description: Financial year deleted successfully
 *       400:
 *         description: Bad request
 */
financialYearRoutes.delete(
  "/delete-financial-year/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 296, "financial_year"),
  controllerWrapper(deleteFinancialYear)
);

export default financialYearRoutes;
