// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { decodeSuperAdminToken } from "../helper/verifyTokenSuperAdmin.js";
import {
  FireEquipmentMaintenanceInspectionCreateUpdate,
  viewFireEquipmentMaintenanceInspection,
} from "../controller/fireEquipmentMaintenanceInspectionController.js";

/**
 * @swagger
 * tags:
 *   name: FireEquipmentMaintenanceInspection
 *   description: API Operations For Managing Fire Equipment Maintenance Inspection
 */
const FireEquipmentMaintenanceInspectionRoutes = Router();

/**
 * @swagger
 * /api/fireEquipmentMaintenanceInspection/create-update-equipment-register:
 *   post:
 *     summary: Create Or Update A Fire Equipment Register
 *     tags: [FireEquipmentRegister]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               equipment_type:
 *                 type: integer
 *                 example: 1
 *               equipment_name:
 *                 type: string
 *                 example: Fire Extinguisher
 *               location:
 *                 type: integer
 *                 example: 9
 *               unique_equipment_id:
 *                 type: integer
 *                 example: 3434
 *               installation_date:
 *                 type: string
 *                 format: date
 *                 example: 2020-10-20
 *               supplier_name:
 *                 type: string
 *                 example: Manufacturer
 *               modal_number:
 *                 type: string
 *                 example: 45
 *               capacity:
 *                 type: string
 *                 example: 1
 *               department:
 *                 type: string
 *                 example:
 *               status:
 *                 type: string
 *                 example: Maintenance Overdue
 *               organization:
 *                 type: integer
 *                 example: 14
 */
FireEquipmentMaintenanceInspectionRoutes.post(
  "/create-update-equipment-maintenance-inspection",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 343, "fire_equipment_maintenance_inspection")(
      req,
      res,
      next
    );
  },
  controllerWrapper(FireEquipmentMaintenanceInspectionCreateUpdate)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/get-equipment-register:
 *   get:
 *     summary: Get list of all Fire Equipment Registers
 *     tags: [FireEquipmentRegister]
 */
FireEquipmentMaintenanceInspectionRoutes.get(
  "/get-equipment-maintenance-inspection",
  verifyToken,
  sideBarPermissionCheck("view", 343, "fire_equipment_maintenance_inspection"),
  controllerWrapper(viewFireEquipmentMaintenanceInspection)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/get-equipment-register/{id}:
 *   get:
 *     summary: Get A Specific Fire Equipment Register record
 *     tags: [FireEquipmentRegister]
 */
FireEquipmentMaintenanceInspectionRoutes.get(
  "/get-equipment-maintenance-inspection/:id",
  verifyToken,
  sideBarPermissionCheck("view", 343, "fire_equipment_maintenance_inspection"),
  controllerWrapper(viewFireEquipmentMaintenanceInspection)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/delete-fire-equipment-register/{id}:
 *   delete:
 *     summary: Delete a specific record of Fire Equipment Register
 *     tags: [FireEquipmentRegister]
 */
// FireEquipmentMaintenanceInspectionRoutes.delete(
//   "/delete-fire-equipment-register/:id",
//   verifyToken,
//   sideBarPermissionCheck("delete", 313, "fire_equipment_register"),
//   controllerWrapper(deleteFireEquipmentRegister)
// );

// FireEquipmentMaintenanceInspectionRoutes.post(
//   "/update-fire-equipment-register-status",
//   verifyToken,
//   sideBarPermissionCheck("edit", 313, "fire_equipment_register"),
//   controllerWrapper(updateFireEquipmentRegisterStatus)
// );

// FireEquipmentMaintenanceInspectionRoutes.get(
//   "/get-fire-equipment-register-status-history",
//   verifyToken,
//   // decodeSuperAdminToken,
//   sideBarPermissionCheck("view", 313, "fire_equipment_register"),
//   controllerWrapper(viewFireEquipmentRegisterStatusHistory)
// );

// FireEquipmentMaintenanceInspectionRoutes.get(
//   "/get-fire-equipment-register-status-history/:id",
//   verifyToken,
//   sideBarPermissionCheck("view", 313, "fire_equipment_register"),
//   controllerWrapper(viewFireEquipmentRegisterStatusHistory)
// );

// FireEquipmentMaintenanceInspectionRoutes.post(
//   "/update-fire-equipment-register-history-status",
//   verifyToken,
//   // sideBarPermissionCheck("edit", 313, "fire_equipment_register"),
//   controllerWrapper(updateFireEquipmentRegisterHistoryStatus)
// );

/**
 * @swagger
 * /api/fireEquipmentRegister/get-fire-equipment-register-timeline/{id}:
 *   get:
 *     summary: Get timeline of Fire Equipment Registers
 *     tags: [FireEquipmentRegister]
 */
// FireEquipmentMaintenanceInspectionRoutes.get(
//   "/get-fire-equipment-register-timeline/:id",
//   verifyToken,
//   // sideBarPermissionCheck("view", 313, "fire_equipment_register"),
//   controllerWrapper(viewFireEquipmentRegisterHistoryTimeline)
// );
export default FireEquipmentMaintenanceInspectionRoutes;
