// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteFlammableStore,
  FlammableStoreCreateUpdate,
  viewFlammableStore,
} from "../controller/flammableStoreController.js";
import { controllerWrapper } from "../helper/wrapper.js";


/**
 * @swagger
 * tags:
 *   name: FlammableStore
 *   description: API Operations For Managing Flammable Store
 */
const FlammableStoreRoutes = Router();


/**
 * @swagger
 * /api/flammableStore/create-update-flammable-store:
 *   post:
 *     summary: Create Or Update A Flammable Store
 *     tags: [FlammableStore]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               equipment_type:
 *                 type: integer
 *                 example: 1
 *               equipment_name:
 *                 type: string
 *                 example: Fire Extinguisher
 *               location:
 *                 type: integer
 *                 example: 9
 *               unique_equipment_id:
 *                 type: integer
 *                 example: 3434
 *               installation_date:
 *                 type: string
 *                 format: date
 *                 example: 2020-10-20
 *               supplier_name:
 *                 type: string
 *                 example: Manufacturer
 *               modal_number:
 *                 type: string
 *                 example: 45
 *               capacity:
 *                 type: string
 *                 example: 1
 *               department:
 *                 type: string
 *                 example: 
 *               status:
 *                 type: string
 *                 example: Maintenance Overdue
 *               organization:
 *                 type: integer
 *                 example: 14
 */
FlammableStoreRoutes.post(
  "/create-update-flammable-store",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      313,
      "flammable_store"
    )(req, res, next);
  },
  controllerWrapper(FlammableStoreCreateUpdate)
);

/**
 * @swagger
 * /api/flammableStore/get-flammable-store:
 *   get:
 *     summary: Get list of all Flammable Store
 *     tags: [FlammableStore]
 */
FlammableStoreRoutes.get(
  "/get-flammable-store",
  verifyToken,
  sideBarPermissionCheck("view", 313, "flammable_store"),
  controllerWrapper(viewFlammableStore)
);

/**
 * @swagger
 * /api/flammableStore/get-flammable-store/{id}:
 *   get:
 *     summary: Get A Specific Flammable Store record
 *     tags: [FlammableStore]
 */
FlammableStoreRoutes.get(
  "/get-flammable-store/:id",
  verifyToken,
  sideBarPermissionCheck("view", 313, "flammable_store"),
  controllerWrapper(viewFlammableStore)
);

/**
 * @swagger
 * /api/flammableStore/delete-flammable-store/{id}:
 *   delete:
 *     summary: Delete a specific record of Flammable Store
 *     tags: [FlammableStore]
 */
FlammableStoreRoutes.delete(
  "/delete-flammable-store/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 313, "flammable_store"),
  controllerWrapper(deleteFlammableStore)
);

export default FlammableStoreRoutes;
