import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createHazmatFireTechnical,
  getHazmatFireTechnical,
  deleteHazmatFireTechnical,
} from "../controller/hazmatFireTechnicalFormController.js";

const hazmatFireTechnicalRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Hazmat Fire Technical Forms
 *   description: API operations for managing hazmat fire technical forms
 */

/**
 * @swagger
 * /api/hazmatFireTechnical/create-hazmat-technical-form:
 *   post:
 *     summary: Create a new hazmat fire technical form entry
 *     tags: [Hazmat Fire Technical Forms]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               incident_id:
 *                 type: integer
 *               category_id:
 *                 type: integer
 *               type_of_occurrence:
 *                 type: string
 *               table_data:
 *                 type: object
 *               location_of_occurrence:
 *                 type: string
 *               date_of_occurrence:
 *                 type: string
 *                 format: date
 *               time_of_occurrence:
 *                 type: string
 *                 format: time
 *               occurrence_ref_no:
 *                 type: string
 *               immediate_cause_of_occurrence:
 *                 type: string
 *               involved_employee_details_name:
 *                 type: string
 *               involved_employee_details_employee_sap:
 *                 type: string
 *               involved_employee_details_employee_type:
 *                 type: string
 *               involved_employee_details_experience:
 *                 type: string
 *               involved_employee_details_role:
 *                 type: string
 *               was_alcohol_or_substance_testing_done:
 *                 type: string
 *               select_substance_testing:
 *                 type: string
 *               tested_by_who:
 *                 type: string
 *               substance_test_results:
 *                 type: string
 *               method_of_testing:
 *                 type: string
 *               was_person_or_operator_licensed_trained:
 *                 type: string
 *               was_license_training_applicable_to_task:
 *                 type: string
 *               license_training_expiry_date:
 *                 type: string
 *                 format: date
 *               license_no_training_certificate_no:
 *                 type: string
 *               year_issued:
 *                 type: integer
 *               issued_by:
 *                 type: string
 *               occurrence_level:
 *                 type: string
 *               was_case_reported_to_authority_regulator:
 *                 type: string
 *               name_of_authority:
 *                 type: string
 *               date_reported_to_authority:
 *                 type: string
 *                 format: date
 *               reported_by_who:
 *                 type: string
 *               estimated_cost_medical_expenses:
 *                 type: number
 *                 format: float
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               signature:
 *                 type: string
 *               date:
 *                 type: string
 *                 format: date
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example:
 *             incident_id: 123
 *             category_id: 456
 *             type_of_occurrence: "Fire"
 *             table_data: {}
 *             location_of_occurrence: "Warehouse"
 *             date_of_occurrence: "2023-12-31"
 *             time_of_occurrence: "14:00"
 *             occurrence_ref_no: "REF123"
 *             immediate_cause_of_occurrence: "Electrical Fault"
 *             involved_employee_details_name: "John Doe"
 *             involved_employee_details_employee_sap: "SAP123"
 *             involved_employee_details_employee_type: "Permanent"
 *             involved_employee_details_experience: "5 years"
 *             involved_employee_details_role: "Technician"
 *             was_alcohol_or_substance_testing_done: "Yes"
 *             select_substance_testing: "Alcohol"
 *             tested_by_who: "Supervisor"
 *             substance_test_results: "Negative"
 *             method_of_testing: "Breathalyzer"
 *             was_person_or_operator_licensed_trained: "Yes"
 *             was_license_training_applicable_to_task: "Yes"
 *             license_training_expiry_date: "2024-01-01"
 *             license_no_training_certificate_no: "LIC12345"
 *             year_issued: 2022
 *             issued_by: "Authority"
 *             occurrence_level: "High"
 *             was_case_reported_to_authority_regulator: "Yes"
 *             name_of_authority: "Fire Department"
 *             date_reported_to_authority: "2023-12-31"
 *             reported_by_who: "John Doe"
 *             estimated_cost_medical_expenses: 5000.00
 *             comments: "Initial report"
 *             form_completed_by: "Jane Smith"
 *             designation: "Manager"
 *             signature: "Signature"
 *             date: "2023-12-31"
 *             organization: 1
 *             department: 2
 *     responses:
 *       '200':
 *         description: Successfully created a new hazmat fire technical form entry
 *       '400':
 *         description: Bad request. Invalid input parameters
 *       '500':
 *         description: Internal server error
 */
hazmatFireTechnicalRoutes.post(
  "/create-hazmat-technical-form",
  verifyToken,
  controllerWrapper(createHazmatFireTechnical)
);

/**
 * @swagger
 * /api/hazmatFireTechnical/get-hazmat-technical-form/{id}:
 *   get:
 *     summary: Get a hazmat fire technical form entry by ID
 *     tags: [Hazmat Fire Technical Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the hazmat fire technical form entry to get
 *     responses:
 *       '200':
 *         description: Successfully retrieved the hazmat fire technical form entry
 *       '404':
 *         description: Hazmat fire technical form entry not found
 *       '500':
 *         description: Internal server error
 */
hazmatFireTechnicalRoutes.get(
  "/get-hazmat-technical-form/:id",
  verifyToken,
  controllerWrapper(getHazmatFireTechnical)
);

/**
 * @swagger
 * /api/hazmatFireTechnical/delete-hazmat-technical-form/{id}:
 *   delete:
 *     summary: Delete a hazmat fire technical form entry by ID
 *     tags: [Hazmat Fire Technical Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the hazmat fire technical form entry to delete
 *     responses:
 *       '200':
 *         description: Successfully deleted the hazmat fire technical form entry
 *       '404':
 *         description: Hazmat fire technical form entry not found
 *       '500':
 *         description: Internal server error
 */
hazmatFireTechnicalRoutes.delete(
  "/delete-hazmat-technical-form/:id",
  verifyToken,
  controllerWrapper(deleteHazmatFireTechnical)
);

export default hazmatFireTechnicalRoutes;
