    // Import necessary modules
    import { Router } from "express";
    import { verifyToken } from "../helper/tokenVerify.js";
    import {
      createUpdateHighestQualification,
      deleteHighestQualification,
      viewHighestQualification,
    } from "../controller/highestQualificationController.js";
    import { sideBarPermissionCheck } from "../helper/checkPermission.js";
    import { controllerWrapper } from "../helper/wrapper.js";
    
    /**
     * @swagger
     * tags:
     *   name: Highest Qualification
     *   description: API operations for managing all Highest Qualification
     */
    const highestQualificationRoutes = Router();
    
    /**
     * @swagger
     * /api/highestQualification/create-update-highest-qualification:
     *   post:
     *     summary: Create or update an Highest Qualification
     *     tags: [Highest Qualification]
     *     requestBody:
     *       required: true
     *       content:
     *         application/json:
     *           schema:
     *             type: object
     *             properties:
     *               id:
     *                 type: string
     *               name:
     *                 type: string
     *               description:
     *                 type: string
     *               organization:
     *                 type: array
     *                 items:
     *                   type: string
     *           example:
     *             id: "id"
     *             name: "Policy Impact Name"
     *             description: "Description of the highest-qualification"
     *             organization: ["Organization1", "Organization2"]
     *     responses:
     *       200:
     *         description: Highest Qualification created or updated successfully
     *       400:
     *         description: Bad request. Invalid input parameters
     *       500:
     *         description: Internal server error
     */
    highestQualificationRoutes.post(
      "/create-update-highest-qualification",
      verifyToken,
      (req, res, next) => {
        sideBarPermissionCheck(req.body.id ? "edit" : "create", 340, "highest_qualification")(
          req,
          res,
          next
        );
      },
      controllerWrapper(createUpdateHighestQualification)
    );
    
    /**
     * @swagger
     * /api/highestQualification/get-highest-qualification:
     *   get:
     *     summary: Get list of all HighestQualification
     *     tags: [Highest Qualification]
     *     responses:
     *       200:
     *         description: List of highestQualification
     *       500:
     *         description: Internal server error
     */
    highestQualificationRoutes.get(
      "/get-highest-qualification",
      verifyToken,
      sideBarPermissionCheck("view", 340, "highest_qualification"),
      controllerWrapper(viewHighestQualification)
    );
    
    /**
     * @swagger
     * /api/highestQualification/get-highest-qualification/{id}:
     *   get:
     *     summary: Get a specific record of Highest Qualification
     *     tags: [Highest Qualification]
     *     parameters:
     *       - in: path
     *         name: id
     *         schema:
     *           type: string
     *         required: true
     *         description: The ID of the highest-qualification
     *     responses:
     *       200:
     *         description: Highest Qualification data
     *       404:
     *         description: Highest Qualification not found
     *       500:
     *         description: Internal server error
     */
    highestQualificationRoutes.get(
      "/get-highest-qualification/:id",
      verifyToken,
      sideBarPermissionCheck("view", 340, "highest_qualification"),
      controllerWrapper(viewHighestQualification)
    );
    
    /**
     * @swagger
     * /api/highestQualification/delete-highest-qualification/{id}:
     *   delete:
     *     summary: Delete a specific record of Highest Qualification
     *     tags: [Highest Qualification]
     *     parameters:
     *       - in: path
     *         name: id
     *         schema:
     *           type: string
     *         required: true
     *         description: The ID of the highest-qualification to delete
     *     responses:
     *       200:
     *         description: Highest Qualification successfully deleted
     *       404:
     *         description: Highest Qualification not found
     *       500:
     *         description: Internal server error
     */
    highestQualificationRoutes.delete(
      "/delete-highest-qualification/:id",
      verifyToken,
      sideBarPermissionCheck("delete", 340, "highest_qualification"),
      controllerWrapper(deleteHighestQualification)
    );
    
    export default highestQualificationRoutes; 
    