import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateInsuranceClaim,
  deleteInsuranceClaim,
  getInsuranceClaim,
  sendInsuranceClaim,
} from "../controller/insuranceClaimRecordingController.js";

const insuranceClaimRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Insurance Claim Recording And Approved Claim Payout Status
 *   description: API operations for managing insurance claims and approved payout status
 */

/**
 * @swagger
 * /api/insuranceClaim/create-update-insurance-claim:
 *   post:
 *     summary: Create or update an insurance claim
 *     tags: [Insurance Claim Recording And Approved Claim Payout Status]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               incident_report_id:
 *                 type: string
 *               description:
 *                 type: string
 *               incident_category:
 *                 type: string
 *               date_of_incident:
 *                 type: string
 *                 format: date
 *               time_of_incident_occurrence:
 *                 type: string
 *               initial_incident_severity:
 *                 type: string
 *               location:
 *                 type: string
 *               insurance_policy_impact:
 *                 type: string
 *               estimated_damage_cost:
 *                 type: number
 *                 format: float
 *               currency:
 *                 type: string
 *               claim_number:
 *                 type: string
 *               actual_damage_cost:
 *                 type: number
 *                 format: float
 *               insurance_claim_approval_status:
 *                 type: string
 *               comments:
 *                 type: string
 *               organization:
 *                 type: string
 *               department:
 *                 type: string
 *           example:
 *             incident_report_id: "INC123"
 *             description: "Description of the insurance claim"
 *             incident_category: "Category of the incident"
 *             date_of_incident: "2023-01-01"
 *             time_of_incident_occurrence: "12:00:00"
 *             initial_incident_severity: "High"
 *             location: "Location of the incident"
 *             insurance_policy_impact: "Impact on insurance policy"
 *             estimated_damage_cost: 5000.00
 *             currency: "USD"
 *             claim_number: "CLM123"
 *             actual_damage_cost: 4500.00
 *             insurance_claim_approval_status: "Pending"
 *             comments: "Additional comments"
 *             organization: "Organization name"
 *             department: "Department name"
 *     responses:
 *       200:
 *         description: Insurance claim created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
insuranceClaimRoutes.post(
  "/create-update-insurance-claim",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      191,
      "insurance_claim"
    )(req, res, next);
  },
  controllerWrapper(createUpdateInsuranceClaim)
);

/**
 * @swagger
 * /api/insuranceClaim/get-insurance-claim/{id}:
 *   get:
 *     summary: Get a specific insurance claim by ID
 *     tags: [Insurance Claim Recording And Approved Claim Payout Status]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the insurance claim
 *     responses:
 *       200:
 *         description: Insurance claim data
 *       404:
 *         description: Insurance claim not found
 *       500:
 *         description: Internal server error
 */
insuranceClaimRoutes.get(
  "/get-insurance-claim/:id",
  verifyToken,
  sideBarPermissionCheck("view", 191, "insurance_claim"),
  controllerWrapper(getInsuranceClaim)
);

/**
 * @swagger
 * /api/insuranceClaim/get-insurance-claim:
 *   get:
 *     summary: Get all insurance claims
 *     tags: [Insurance Claim Recording And Approved Claim Payout Status]
 *     responses:
 *       200:
 *         description: List of insurance claims
 *       500:
 *         description: Internal server error
 */
insuranceClaimRoutes.get(
  "/get-insurance-claim",
  verifyToken,
  sideBarPermissionCheck("view", 191, "insurance_claim"),
  controllerWrapper(getInsuranceClaim)
);

/**
 * @swagger
 * /api/insuranceClaim/delete-insurance-claim/{id}:
 *   delete:
 *     summary: Delete a specific insurance claim by ID
 *     tags: [Insurance Claim Recording And Approved Claim Payout Status]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the insurance claim to delete
 *     responses:
 *       200:
 *         description: Insurance claim successfully deleted
 *       404:
 *         description: Insurance claim not found
 *       500:
 *         description: Internal server error
 */
insuranceClaimRoutes.delete(
  "/delete-insurance-claim/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 191, "insurance_claim"),
  controllerWrapper(deleteInsuranceClaim)
);

/**
 * @swagger
 * /api/insuranceClaim/send-mail:
 *   post:
 *     summary: Send mail for insurance claim
 *     tags: [Insurance Claim Recording And Approved Claim Payout Status]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "1"
 *             mails: "[email1, email2]"
 *             url: "url"
 *     responses:
 *       200:
 *         description: Insurance claim send successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */

insuranceClaimRoutes.post(
  "/send-mail",
  // verifyToken,
  controllerWrapper(sendInsuranceClaim)
);

export default insuranceClaimRoutes;
