// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateInsurancePolicyImpact,
  deleteInsurancePolicyImpact,
  viewInsurancePolicyImpact,
} from "../controller/insurancePolicyImpactController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Insurance Policy Impact
 *   description: API operations for managing all Insurance Policy Impact
 */
const insurancePolicyImpactRoutes = Router();

/**
 * @swagger
 * /api/insurancePolicyImpact/create-update-insurance-policy-impact:
 *   post:
 *     summary: Create or update an Insurance Policy Impact
 *     tags: [Insurance Policy Impact]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the insurance policy impact"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Insurance policy impact created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
insurancePolicyImpactRoutes.post(
  "/create-update-insurance-policy-impact",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      198,
      "insurance_policy_impact"
    )(req, res, next);
  },
  controllerWrapper(createUpdateInsurancePolicyImpact)
);

/**
 * @swagger
 * /api/insurancePolicyImpact/get-insurance-policy-impact:
 *   get:
 *     summary: Get list of all Insurance Policy Impacts
 *     tags: [Insurance Policy Impact]
 *     responses:
 *       200:
 *         description: List of insurance policy impacts
 *       500:
 *         description: Internal server error
 */
insurancePolicyImpactRoutes.get(
  "/get-insurance-policy-impact",
  verifyToken,
  sideBarPermissionCheck("view", 198, "insurance_policy_impact"),
  controllerWrapper(viewInsurancePolicyImpact)
);

/**
 * @swagger
 * /api/insurancePolicyImpact/get-insurance-policy-impact/{id}:
 *   get:
 *     summary: Get a specific record of Insurance Policy Impact
 *     tags: [Insurance Policy Impact]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the insurance policy impact
 *     responses:
 *       200:
 *         description: Insurance policy impact data
 *       404:
 *         description: Insurance policy impact not found
 *       500:
 *         description: Internal server error
 */
insurancePolicyImpactRoutes.get(
  "/get-insurance-policy-impact/:id",
  verifyToken,
  sideBarPermissionCheck("view", 198, "insurance_policy_impact"),
  controllerWrapper(viewInsurancePolicyImpact)
);

/**
 * @swagger
 * /api/insurancePolicyImpact/delete-insurance-policy-impact/{id}:
 *   delete:
 *     summary: Delete a specific record of Insurance Policy Impact
 *     tags: [Insurance Policy Impact]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the insurance policy impact to delete
 *     responses:
 *       200:
 *         description: Insurance policy impact successfully deleted
 *       404:
 *         description: Insurance policy impact not found
 *       500:
 *         description: Internal server error
 */
insurancePolicyImpactRoutes.delete(
  "/delete-insurance-policy-impact/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 198, "insurance_policy_impact"),
  controllerWrapper(deleteInsurancePolicyImpact)
);

export default insurancePolicyImpactRoutes;
