// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedLicense,
  deleteLicense,
  viewAllLicense,
} from "../controller/licenseController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { licenseCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: License
 *   description: API operations for managing all licenses
 */
const licenseRoutes = Router();

/**
 * @swagger
 * /api/license/create-update-license:
 *   post:
 *     summary: Create or Update a License
 *     tags: [License]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             name: "License Name"
 *             description: "License Description"
 *             organization: [1, 2]
 *     responses:
 *       200:
 *         description: License created or updated successfully
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
licenseRoutes.post(
  "/create-update-license",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 136, "license")(
      req,
      res,
      next
    );
  },
  validateRequest(licenseCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedLicense)
);

/**
 * @swagger
 * /api/license/get-license:
 *   get:
 *     summary: Get list of all licenses
 *     tags: [License]
 *     responses:
 *       200:
 *         description: A list of licenses
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
licenseRoutes.get(
  "/get-license/",
  verifyToken,
  sideBarPermissionCheck("view", 136, "license"),
  controllerWrapper(viewAllLicense)
);

/**
 * @swagger
 * /api/license/get-license/{id}:
 *   get:
 *     summary: Get a specific record of license
 *     tags: [License]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the license
 *     responses:
 *       200:
 *         description: A specific license record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
licenseRoutes.get(
  "/get-license/:id",
  verifyToken,
  sideBarPermissionCheck("view", 136, "license"),
  controllerWrapper(viewAllLicense)
);

/**
 * @swagger
 * /api/license/delete-license/{id}:
 *   delete:
 *     summary: Delete a specific record of license
 *     tags: [License]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the license to delete
 *     responses:
 *       200:
 *         description: License deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
licenseRoutes.delete(
  "/delete-license/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 136, "license"),
  controllerWrapper(deleteLicense)
);

export default licenseRoutes;
