// Import necessary modules
import { Router } from "express";
import {
  createUpdateObjectiveSetting,
  viewAllObjectiveSetting,
  deleteObjectiveSetting,
  viewAllObjectiveTargetManagementPlan,
  updateApprovalStatus,
  updateActionProgressStatus,
  viewAllApprovalWorkflowObjectiveSetting,
} from "../controller/objectiveSettingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { objectiveSettingCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: ObjectiveSetting
 *   description: API operations for managing all ObjectiveSetting
 */
const ObjectiveSettingRoutes = Router();

/**
 * @swagger
 * /api/ObjectiveSetting/create-update-objective-setting:
 *   post:
 *     summary: Create or update an Objective Setting
 *     tags:
 *       - ObjectiveSetting
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             organization: 2
 *             parent_objective: ""
 *             associated_risks: ""
 *             objective_target_id: ""
 *             objective_name: ""
 *             objective_owner: ""
 *             date_generated: ""
 *             objective_description: ""
 *             human_resource_requirements:
 *               - role: ""
 *                 no_of_people: ""
 *             technological_resources:
 *               - resource: ""
 *             resource_type: "day"
 *             amount: ""
 *             target_id: ""
 *             measurement_metric: ""
 *             numeric_value: ""
 *             target_name: ""
 *             kpi_name: ""
 *             target_description: ""
 *             management_plan_name: ""
 *             management_plan_description: ""
 *             management_plan_actions:
 *               - action_id: ""
 *                 action_description: ""
 *                 responsible_department: 3
 *                 responsible_person: ""
 *                 start_date: ""
 *                 due_date: ""
 *                 action_progress_description: ""
 *                 action_progress_status: ""
 */
ObjectiveSettingRoutes.post(
  "/create-update-objective-setting",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 76, "objective_setting")(req, res, next);
  },
  validateRequest(objectiveSettingCreateUpdateValidationSchema),
  controllerWrapper(createUpdateObjectiveSetting)
);

/**
 * @swagger
 * /api/ObjectiveSetting/get-objective-setting:
 *   get:
 *     summary: Get list of all Objective Setting
 *     tags: [ObjectiveSetting]
 */
ObjectiveSettingRoutes.get(
  "/get-objective-setting",
  verifyToken,
  sideBarPermissionCheck("view", 76, "objective_setting"),
  controllerWrapper(viewAllObjectiveSetting)
);

/**
 * @swagger
 * /api/ObjectiveSetting/get-objective-setting/{id}:
 *   get:
 *     summary: Get a specific record of Objective Setting
 *     tags: [ObjectiveSetting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Successfully retrieved Objective Setting
 *       404:
 *         description: Objective Setting not found
 */
ObjectiveSettingRoutes.get(
  "/get-objective-setting/:id",
  verifyToken,
  sideBarPermissionCheck("view", 76, "objective_setting"),
  controllerWrapper(viewAllObjectiveSetting)
);

/**
 * @swagger
 * /api/ObjectiveSetting/get-objective-setting-approval-workflow:
 *   get:
 *     summary: Get list of all Objective Setting Approval Workflow
 *     tags: [ObjectiveSetting]
 */
ObjectiveSettingRoutes.get(
  "/get-objective-setting-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 76, "objective_setting"),
  controllerWrapper(viewAllApprovalWorkflowObjectiveSetting)
);
/**
 * @swagger
 * /api/ObjectiveSetting/get-objective-setting-approval-workflow/{id}:
 *   get:
 *     summary: Get a specific record of Objective Setting Approval Workflow
 *     tags: [ObjectiveSetting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Successfully retrieved Objective Setting
 *       404:
 *         description: Objective Setting not found
 */
ObjectiveSettingRoutes.get(
  "/get-objective-setting-approval-workflow/:id",
  verifyToken,
  sideBarPermissionCheck("view", 76, "objective_setting"),
  controllerWrapper(viewAllApprovalWorkflowObjectiveSetting)
);
/**
 * @swagger
 * /api/ObjectiveSetting/delete-objective-setting/{id}:
 *   delete:
 *     summary: Delete a specific record of Objective Setting
 *     tags: [ObjectiveSetting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Successfully deleted Objective Setting
 *       404:
 *         description: Objective Setting not found
 */
ObjectiveSettingRoutes.delete(
  "/delete-objective-setting/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 76, "objective_setting"),
  controllerWrapper(deleteObjectiveSetting)
);

/**
 * @swagger
 * /api/ObjectiveSetting/get-objective-setting-target-plan:
 *   get:
 *     summary: Get list of all Objective Setting Target Management Plans
 *     tags: [ObjectiveSetting]
 */
ObjectiveSettingRoutes.get(
  "/get-objective-setting-target-plan",
  verifyToken,
  sideBarPermissionCheck("view", 76, "objective_setting"),
  controllerWrapper(viewAllObjectiveTargetManagementPlan)
);

/**
 * @swagger
 * /api/ObjectiveSetting/approve-objective-setting-management-plan:
 *   post:
 *     summary: Approve an Objective Setting Management Plan
 *     tags: [ObjectiveSetting]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "123"
 *             approval_status: "approved"  # Clarify the purpose
 */
ObjectiveSettingRoutes.post(
  "/approve-objective-setting-management-plan",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 76, "objective_setting")(req, res, next);
  },
  controllerWrapper(updateActionProgressStatus)
);
/**
 * @swagger
 * /api/ObjectiveSetting/update-objective-setting-status:
 *   post:
 *     summary: Approve an Objective Setting
 *     tags: [ObjectiveSetting]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "123"
 *             status: "Approved"  # Clarify the purpose
 */
ObjectiveSettingRoutes.post(
  "/update-objective-setting-status",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 76, "objective_setting")(req, res, next);
  },
  controllerWrapper(updateApprovalStatus)
);

export default ObjectiveSettingRoutes;
