import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateOhsFileAssessment,
  getOhsFileAssessment,
  deleteOhsFileAssessment,
} from "../controller/ohsFileAssessmentController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor PreSite
 *   description: API operations for managing ohs file assessment
 */
/**
 * @swagger
 * components:
 *   schemas:
 *     OhsFileAssessment:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the OHS file assessment.
 *         ohs_requirement:
 *           type: integer
 *           description: The ohs requirement ID.
 *         form:
 *           type : string
 *           description : The form data in the long text format.
 *         organization:
 *           type: integer
 *           description: The associated organization ID.
 *         department:
 *           type: integer
 *           description: The associated department ID.
 *         deleted:
 *           type: integer
 *           description: Whether the OHS file is marked as deleted (0 or 1).
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the OHS file.
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The timestamp when the OHS file was created.
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the OHS file.
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The timestamp when the OHS file was last updated.
 */

const ohsFileAssessmentRoutes = Router();
/**
 * @swagger
 * /api/ohsFileAssessment/create-update-ohs-file-assessment:
 *   post:
 *     summary: Create or update an OHS file assessment.
 *     tags: [Ohs File Assessment]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OhsFileAssessment'
 *     responses:
 *       200:
 *         description: The OHS file assessment was successfully created/updated.
 */

ohsFileAssessmentRoutes.post(
  "/create-update-ohs-file-assessment",
  verifyToken,
  controllerWrapper(createUpdateOhsFileAssessment)
);

/**
 * @swagger
 * /api/ohsFileAssessment/get-ohs-file-assessment:
 *   get:
 *     summary: Retrieve all OHS file assessments.
 *     tags: [Ohs File Assessment]
 *     responses:
 *       200:
 *         description: A list of OHS file assessments.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OhsFileAssessment'
 */

ohsFileAssessmentRoutes.get(
  "/get-ohs-file-assessment",
  verifyToken,
  controllerWrapper(getOhsFileAssessment)
);

/**
 * @swagger
 * /api/ohsFileAssessment/get-ohs-file-assessment/{id}:
 *   get:
 *     summary: Get an OHS file assessment by ID.
 *     tags: [Ohs File Assessment]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the OHS file assessment.
 *     responses:
 *       200:
 *         description: An OHS file assessment.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OhsFileAssessment'
 */

ohsFileAssessmentRoutes.get(
  "/get-ohs-file-assessment/:id",
  verifyToken,
  controllerWrapper(getOhsFileAssessment)
);

/**
 * @swagger
 * /api/ohsFileAssessment/delete-ohs-file-assessment/{id}:
 *   delete:
 *     summary: Delete an OHS file assessment by ID.
 *     tags: [Ohs File Assessment]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the OHS file assessment to delete.
 *     responses:
 *       200:
 *         description: The OHS file assessment was deleted successfully.
 */

ohsFileAssessmentRoutes.delete(
  "/delete-ohs-file-assessment/:id",
  verifyToken,
  controllerWrapper(deleteOhsFileAssessment)
);

export default ohsFileAssessmentRoutes;
