// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedPermitLicense,
  deletePermitLicense,
  viewAllPermitLicense,
} from "../controller/permitLicenseController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {  validateRequest } from "../helper/general.js";
import { permitLicenseCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
/**
 * @swagger
 * tags:
 *   name: Permit License
 *   description: API operations for managing all Permit Licenses
 */
const permitLicenseRoutes = Router();

/**
 * @swagger
 * /api/permitLicense/create-update-permit-license:
 *   post:
 *     summary: Create or update a Permit License
 *     tags: [Permit License]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "1"
 *             name: "Environmental Permit"
 *             description: "Permit for operating a waste management facility"
 *             organization: ["ABC Waste Management Inc."]
 */
permitLicenseRoutes.post(
  "/create-update-permit-license",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      148,
      "permit_license"
    )(req, res, next);
  },
  validateRequest(permitLicenseCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/get-permit-license:
 *   get:
 *     summary: Get list of all Permit Licenses
 *     tags: [Permit License]
 */
permitLicenseRoutes.get(
  "/get-permit-license",
  verifyToken,
  sideBarPermissionCheck("view", 148, "permit_license"),
  controllerWrapper(viewAllPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/get-permit-license/{id}:
 *   get:
 *     summary: Get a specific record of Permit License
 *     tags: [Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Permit License
 */
permitLicenseRoutes.get(
  "/get-permit-license/:id",
  verifyToken,
  sideBarPermissionCheck("view", 148, "permit_license"),
  controllerWrapper(viewAllPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/delete-permit-license/{id}:
 *   delete:
 *     summary: Delete a specific record of Permit License
 *     tags: [Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Permit License
 */
permitLicenseRoutes.delete(
  "/delete-permit-license/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 148, "permit_license"),
  controllerWrapper(deletePermitLicense)
);

export default permitLicenseRoutes;
