import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateProjectRegistration,
  deleteProjectRegistration,
  getProjectRegistration,
} from "../controller/projectRegistrationController.js";

/**
 * @swagger
 * tags:
 *   name: Project Registration
 *   description: API operations for managing project registrations
 */

const projectRegistrationRoutes = Router();

/**
 * @swagger
 * /api/projectRegistration/create-update-project-registration:
 *   post:
 *     summary: Create or update a project registration
 *     tags: [Project Registration]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             contractor: 1
 *             cipc_registration_number: "CIPC12345"
 *             project_title: "New Project Title"
 *             project_description: "Description of the project"
 *             type_of_work: "Construction"
 *             project_start_date: "2024-10-01"
 *             project_end_date: "2024-12-31"
 *             project_value: 1000000
 *             contract_value: 800000
 *             project_location: "Location of the project"
 *             currency: "USD"
 *             classification_value: "High"
 *             classification_criticality: "Critical"
 *             classification_impact: "Major impact"
 *             classification_category: "Category A"
 *             project_members: "Member 1, Member 2"
 *             sub_contractor: 2
 *             sub_contractors_personnel: "Personnel 1, Personnel 2"
 *     responses:
 *       200:
 *         description: Project registration created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
projectRegistrationRoutes.post(
  "/create-update-project-registration",
  verifyToken,
  controllerWrapper(createUpdateProjectRegistration)
);

/**
 * @swagger
 * /api/projectRegistration/get-project-registration:
 *   get:
 *     summary: Get all project registrations
 *     tags: [Project Registration]
 *     responses:
 *       200:
 *         description: List of all project registrations
 *       500:
 *         description: Internal server error
 */
projectRegistrationRoutes.get(
  "/get-project-registration",
  verifyToken,
  controllerWrapper(getProjectRegistration)
);

/**
 * @swagger
 * /api/projectRegistration/get-project-registration/{id}:
 *   get:
 *     summary: Get a specific project registration by ID
 *     tags: [Project Registration]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the project registration
 *     responses:
 *       200:
 *         description: Project registration data
 *       404:
 *         description: Project registration not found
 *       500:
 *         description: Internal server error
 */
projectRegistrationRoutes.get(
  "/get-project-registration/:id",
  verifyToken,
  controllerWrapper(getProjectRegistration)
);

/**
 * @swagger
 * /api/projectRegistration/delete-project-registration/{id}:
 *   delete:
 *     summary: Delete a specific project registration by ID
 *     tags: [Project Registration]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the project registration to delete
 *     responses:
 *       200:
 *         description: Project registration successfully deleted
 *       404:
 *         description: Project registration not found
 *       500:
 *         description: Internal server error
 */
projectRegistrationRoutes.delete(
  "/delete-project-registration/:id",
  verifyToken,
  controllerWrapper(deleteProjectRegistration)
);

export default projectRegistrationRoutes;
