import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateSurveyResponse,
  getSurveyResponse,
  deleteSurveyResponse,
} from "../controller/surveyResponseController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const surveyResponseRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Survey Responses
 *   description: API operations for managing survey responses
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     SurveyResponse:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the survey response
 *         survey_id:
 *           type: integer
 *           description: ID of the survey
 *         user_id:
 *           type: integer
 *           description: ID of the user who responded
 *         name:
 *           type: string
 *           description: Name of the respondent
 *         email:
 *           type: string
 *           description: Email of the respondent
 *         phone:
 *           type: string
 *           description: Phone number of the respondent
 *         values:
 *           type: string
 *           description: The response values in JSON format
 *         total:
 *           type: integer
 *           description: Total score of the survey response
 *         deleted:
 *           type: integer
 *           description: Soft delete flag
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: Creation timestamp
 *       example:
 *         id: 1
 *         survey_id: 101
 *         user_id: 202
 *         name: John Doe
 *         email: john.doe@example.com
 *         phone: "1234567890"
 *         values: '{"question1": "answer1", "question2": "answer2"}'
 *         total: 100
 *         deleted: 0
 *         created_at: "2024-07-22T14:00:00Z"
 */

/**
 * @swagger
 * /api/surveyResponse/create-update-survey-response:
 *   post:
 *     summary: Create or update a survey response
 *     tags: [Survey Responses]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             survey_id: 101
 *             user_id: 202
 *             name: John Doe
 *             email: john.doe@example.com
 *             phone: "1234567890"
 *             values: '{"question1": "answer1", "question2": "answer2"}'
 *     responses:
 *       200:
 *         description: The created/updated survey response
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SummaryResponse'
 */
surveyResponseRoutes.post(
  "/create-update-survey-response",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 224, "survey_response")(
      req,
      res,
      next
    )
  },
  controllerWrapper(createUpdateSurveyResponse)
);

/**
 * @swagger
 * /api/surveyResponse/get-survey-response:
 *   get:
 *     summary: Get all survey responses
 *     tags: [Survey Responses]
 *     responses:
 *       200:
 *         description: List of all survey responses
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/SummaryResponse'
 */
surveyResponseRoutes.get(
  "/get-survey-response",
  verifyToken,
  sideBarPermissionCheck("view", "224", "survey_response"),
  controllerWrapper(getSurveyResponse)
);

/**
 * @swagger
 * /api/surveyResponse/get-survey-response/{id}:
 *   get:
 *     summary: Get a survey response by id
 *     tags: [Survey Responses]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The survey response id
 *     responses:
 *       200:
 *         description: The survey response by id
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SummaryResponse'
 */
surveyResponseRoutes.get(
  "/get-survey-response/:id",
  verifyToken,
  sideBarPermissionCheck("view", "224", "survey_response"),
  controllerWrapper(getSurveyResponse)
);

/**
 * @swagger
 *   delete:
 *     summary: Delete a survey response by id
 *     tags: [Survey Responses]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The survey response id
 *     responses:
 *       200:
 *         description: Survey response deleted
 */
surveyResponseRoutes.delete(
  "/delete-survey-response/:id",
  verifyToken,
  sideBarPermissionCheck("delete", "224", "survey_response"),
  controllerWrapper(deleteSurveyResponse)
);

export default surveyResponseRoutes;
