// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteTacticalRisk,
  tacticalRiskCreateUpdate,
  viewTacticalRisk,
} from "../controller/tacticalRiskController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { strategicRiskValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: tacticalRisk
 *   description: API Operations For Managing strategic risk
 */
const tacticalRiskRoutes = Router();

/**
 * @swagger
 * /api/tacticalRisk/create-update-tactical-risk:
 *   post:
 *     summary: Create Or Update A tactical Risk
 *     tags: [tacticalRisk]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id:id,risk_register_name:risk_register_name,assessment_date:assessment_date,review_date:review_date,risk_context:risk_context,risk_assessment_facilitator:risk_assessment_facilitator,risk_register_approver:risk_register_approver,meeting_id:meeting_id,organization:organization,department:department
 */

tacticalRiskRoutes.post(
  "/create-update-tactical-risk",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      44,
      "tactical_risk"
    )(req, res, next);
  },
  validateRequest(strategicRiskValidationSchema),
  controllerWrapper(tacticalRiskCreateUpdate)
);

/**
 * @swagger
 * api/tacticalRisk/get-tactical-risk:
 *   get:
 *     summary: Get list of all strategic risk List
 *     tags: [tacticalRisk]
 */
tacticalRiskRoutes.get(
  "/get-tactical-risk",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk"),
  controllerWrapper(viewTacticalRisk)
);

/**
 * @swagger
 * api/tacticalRisk/get-tactical-risk/{id}:
 *   get:
 *     summary: Get A Specific strategic risk record
 *     tags: [tacticalRisk]
 */
tacticalRiskRoutes.get(
  "/get-tactical-risk/:id",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk"),
  controllerWrapper(viewTacticalRisk)
);

/**
 * @swagger
 * api/tacticalRisk/delete-tactical-risk/{id}:
 *   delete:
 *     summary: delete a specific record of strategic risk
 *     tags: [tacticalRisk]
 */
tacticalRiskRoutes.delete(
  "/delete-tactical-risk/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 44, "tactical_risk"),
  controllerWrapper(deleteTacticalRisk)
);

export default tacticalRiskRoutes;
